<?php

declare(strict_types=1);

namespace Drupal\Tests\prometheus_metrics_commerce\Kernel;

use Drupal\commerce_product\Entity\Product as ProductEntity;
use Drupal\commerce_product\Entity\ProductType;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\prometheus_metrics_commerce\Metrics\Product;
use Drupal\Tests\commerce\Kernel\CommerceKernelTestBase;
use PHPUnit\Framework\Attributes\Group;
use Prometheus\Gauge;

/**
 * Tests the Product metric class.
 */
#[Group("prometheus_metrics_commerce")]
class ProductTest extends CommerceKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'commerce_product',
    'prometheus_metrics',
    'prometheus_metrics_commerce',
  ];

  /**
   * The Product metric service.
   *
   * @var \Drupal\prometheus_metrics_commerce\Metrics\Product
   */
  protected Product $productMetric;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('commerce_product');
    $this->installEntitySchema('commerce_product_variation');
    $this->installConfig(['commerce_product']);

    ProductType::create(
          [
            'id' => 'book',
            'label' => 'Book',
            'variationType' => 'default',
          ]
      )->save();

    // Set site name for testing.
    $this->config('system.site')->set('name', 'Test Site')->save();

    // Set up prometheus_metrics configuration.
    $this->config('prometheus_metrics.configuration')->set('metrics_namespace', 'drupal')->save();

    $this->productMetric = $this->container->get(Product::class);
  }

  /**
   * Tests rebuild() method counts published products by type.
   */
  public function testRebuild(): void {
    // Create published products of different types.
    $this->createProduct('default', TRUE);
    $this->createProduct('default', TRUE);
    $this->createProduct('book', TRUE);

    // Create unpublished products (should not be counted).
    $this->createProduct('default', FALSE);
    $this->createProduct('book', FALSE);

    // Mock the Prometheus gauge to capture set() calls.
    $gaugeCalls = [];
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->exactly(2))
      ->method('set')
      ->willReturnCallback(
              function ($count, $labels) use (&$gaugeCalls) {
                  $gaugeCalls[] = ['count' => $count, 'labels' => $labels];
              }
          );

    // Mock the Prometheus metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->with(
              'drupal',
              'commerce_products_total',
              'Total number of published commerce products',
              ['type', 'site_name']
          )
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->productMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->productMetric, $metricsService);

    // Run rebuild.
    $this->productMetric->rebuild();

    // Verify the correct counts were set.
    $this->assertCount(2, $gaugeCalls);

    // Sort by type for consistent assertion.
    usort($gaugeCalls, fn($a, $b) => strcmp($a['labels'][0], $b['labels'][0]));

    $this->assertEquals(1, $gaugeCalls[0]['count']);
    $this->assertEquals(['book', 'Test Site'], $gaugeCalls[0]['labels']);

    $this->assertEquals(2, $gaugeCalls[1]['count']);
    $this->assertEquals(['default', 'Test Site'], $gaugeCalls[1]['labels']);
  }

  /**
   * Tests incrementFromProduct() increments gauge for published products.
   */
  public function testIncrementFromProduct(): void {
    $product = $this->createProduct('default', TRUE);

    // Mock the Prometheus gauge.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('inc')
      ->with(['default', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->productMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->productMetric, $metricsService);

    $this->productMetric->incrementFromProduct($product);
  }

  /**
   * Tests incrementFromProduct() does nothing for unpublished products.
   */
  public function testIncrementFromProductUnpublished(): void {
    $product = $this->createProduct('default', FALSE);

    // Mock the metrics service - should not be called.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->never())
      ->method('getGauge');

    // Replace the service.
    $reflection = new \ReflectionClass($this->productMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->productMetric, $metricsService);

    $this->productMetric->incrementFromProduct($product);
  }

  /**
   * Tests decrementFromProduct() decrements gauge for published products.
   */
  public function testDecrementFromProduct(): void {
    $product = $this->createProduct('book', TRUE);

    // Mock the Prometheus gauge.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('dec')
      ->with(['book', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->productMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->productMetric, $metricsService);

    $this->productMetric->decrementFromProduct($product);
  }

  /**
   * Tests updateFromProducts() when publish status changes.
   */
  public function testUpdateFromProductsPublishStatusChanged(): void {
    $oldProduct = $this->createProduct('default', FALSE);
    $newProduct = clone $oldProduct;
    $newProduct->setPublished(TRUE);

    // Mock the Prometheus gauge.
    $mockGauge = $this->createMock(Gauge::class);
    $mockGauge->expects($this->once())
      ->method('inc')
      ->with(['default', 'Test Site']);

    // Mock the metrics service.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->once())
      ->method('getGauge')
      ->willReturn($mockGauge);

    // Replace the service.
    $reflection = new \ReflectionClass($this->productMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->productMetric, $metricsService);

    $this->productMetric->updateFromProducts($oldProduct, $newProduct);
  }

  /**
   * Tests updateFromProducts() does nothing when status unchanged.
   */
  public function testUpdateFromProductsNoStatusChange(): void {
    $oldProduct = $this->createProduct('default', TRUE);
    $newProduct = clone $oldProduct;
    $newProduct->setTitle('Updated Title');

    // Mock the metrics service - should not be called.
    $metricsService = $this->createMock('Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface');
    $metricsService->expects($this->never())
      ->method('getGauge');

    // Replace the service.
    $reflection = new \ReflectionClass($this->productMetric);
    $property = $reflection->getProperty('metricsService');
    $property->setAccessible(TRUE);
    $property->setValue($this->productMetric, $metricsService);

    $this->productMetric->updateFromProducts($oldProduct, $newProduct);
  }

  /**
   * Helper method to create a product with a variation.
   *
   * @param string $type
   *   The product type.
   * @param bool $published
   *   Whether the product should be published.
   *
   * @return \Drupal\commerce_product\Entity\ProductInterface
   *   The created product.
   */
  protected function createProduct(string $type, bool $published) {
    $variation = ProductVariation::create(
          [
            'type' => 'default',
            'sku' => $this->randomString(),
            'title' => $this->randomString(),
            'status' => 1,
          ]
      );
    $variation->save();

    $product = ProductEntity::create(
          [
            'type' => $type,
            'title' => $this->randomString(),
            'status' => $published ? 1 : 0,
            'variations' => [$variation],
          ]
      );
    $product->save();

    return $product;
  }

}
