<?php

declare(strict_types=1);

namespace Drupal\Tests\prometheus_metrics_commerce\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Drupal\prometheus_metrics_commerce\Metrics\CommerceMetricBase;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the CommerceMetricBase abstract class.
 */
#[Group("prometheus_metrics_commerce")]
class CommerceMetricBaseTest extends UnitTestCase {

  /**
   * Tests namespace and site name caching with default values.
   */
  public function testDefaultConfiguration(): void {
    $metricsService = $this->createMock(PrometheusMetricsInterface::class);

    $prometheusConfig = $this->createMock(ImmutableConfig::class);
    $prometheusConfig->expects($this->once())
      ->method('get')
      ->with('metrics_namespace')
      ->willReturn(NULL);

    $siteConfig = $this->createMock(ImmutableConfig::class);
    $siteConfig->expects($this->once())
      ->method('get')
      ->with('name')
      ->willReturn(NULL);

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->expects($this->exactly(2))
      ->method('get')
      ->willReturnMap(
              [
              ['prometheus_metrics.configuration', $prometheusConfig],
              ['system.site', $siteConfig],
              ]
          );

    $metric = new TestCommerceMetric($metricsService, $configFactory);

    $this->assertEquals('drupal', $metric->namespaceForTest());
    $this->assertEquals('default', $metric->siteNameForTest());
  }

  /**
   * Tests namespace and site name caching with custom values.
   */
  public function testCustomConfiguration(): void {
    $metricsService = $this->createMock(PrometheusMetricsInterface::class);

    $prometheusConfig = $this->createMock(ImmutableConfig::class);
    $prometheusConfig->expects($this->once())
      ->method('get')
      ->with('metrics_namespace')
      ->willReturn('custom_namespace');

    $siteConfig = $this->createMock(ImmutableConfig::class);
    $siteConfig->expects($this->once())
      ->method('get')
      ->with('name')
      ->willReturn('My Custom Site');

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->expects($this->exactly(2))
      ->method('get')
      ->willReturnMap(
              [
              ['prometheus_metrics.configuration', $prometheusConfig],
              ['system.site', $siteConfig],
              ]
          );

    $metric = new TestCommerceMetric($metricsService, $configFactory);

    $this->assertEquals('custom_namespace', $metric->namespaceForTest());
    $this->assertEquals('My Custom Site', $metric->siteNameForTest());
  }

  /**
   * Tests that values are cached and config is only read once.
   */
  public function testConfigurationCaching(): void {
    $metricsService = $this->createMock(PrometheusMetricsInterface::class);

    $prometheusConfig = $this->createMock(ImmutableConfig::class);
    $prometheusConfig->expects($this->once())
      ->method('get')
      ->with('metrics_namespace')
      ->willReturn('test_namespace');

    $siteConfig = $this->createMock(ImmutableConfig::class);
    $siteConfig->expects($this->once())
      ->method('get')
      ->with('name')
      ->willReturn('Test Site');

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->expects($this->exactly(2))
      ->method('get')
      ->willReturnMap(
              [
              ['prometheus_metrics.configuration', $prometheusConfig],
              ['system.site', $siteConfig],
              ]
          );

    $metric = new TestCommerceMetric($metricsService, $configFactory);

    // Call methods multiple times to verify caching.
    $this->assertEquals('test_namespace', $metric->namespaceForTest());
    $this->assertEquals('test_namespace', $metric->namespaceForTest());
    $this->assertEquals('Test Site', $metric->siteNameForTest());
    $this->assertEquals('Test Site', $metric->siteNameForTest());
  }

}

/**
 * Concrete implementation of CommerceMetricBase for testing.
 */
class TestCommerceMetric extends CommerceMetricBase {

  /**
   * {@inheritdoc}
   */
  public function rebuild(): void {
    // Not tested here.
  }

  /**
   * Helper to expose the protected namespace getter for tests.
   */
  public function namespaceForTest(): string {
    return $this->getNamespace();
  }

  /**
   * Helper to expose the protected site name getter for tests.
   */
  public function siteNameForTest(): string {
    return $this->getSiteName();
  }

}
