<?php

declare(strict_types=1);

namespace Drupal\Tests\prometheus_metrics_commerce\Unit;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Event\OrderEvent;
use Drupal\commerce_order\Event\OrderEvents;
use Drupal\prometheus_metrics_commerce\EventSubscriber\OrderMetricsSubscriber;
use Drupal\prometheus_metrics_commerce\Metrics\OrderRevenue;
use Drupal\prometheus_metrics_commerce\Metrics\OrderTotal;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;

/**
 * Tests the OrderMetricsSubscriber event subscriber.
 */
#[Group("prometheus_metrics_commerce")]
class OrderMetricsSubscriberTest extends UnitTestCase {

  /**
   * Tests getSubscribedEvents returns correct event mapping.
   */
  public function testGetSubscribedEvents(): void {
    $events = OrderMetricsSubscriber::getSubscribedEvents();

    $this->assertArrayHasKey(OrderEvents::ORDER_INSERT, $events);
    $this->assertEquals(['onOrderInsert'], $events[OrderEvents::ORDER_INSERT]);

    $this->assertArrayHasKey(OrderEvents::ORDER_UPDATE, $events);
    $this->assertEquals(['onOrderUpdate'], $events[OrderEvents::ORDER_UPDATE]);

    $this->assertArrayHasKey(OrderEvents::ORDER_DELETE, $events);
    $this->assertEquals(['onOrderDelete'], $events[OrderEvents::ORDER_DELETE]);
  }

  /**
   * Tests onOrderInsert calls both metric services.
   */
  public function testOnOrderInsert(): void {
    $order = $this->createMock(OrderInterface::class);

    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('incrementFromOrder')
      ->with($order);

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->once())
      ->method('incrementFromOrder')
      ->with($order);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->never())->method('log');

    $event = $this->createMock(OrderEvent::class);
    $event->expects($this->once())
      ->method('getOrder')
      ->willReturn($order);

    $subscriber = new OrderMetricsSubscriber($orderTotal, $orderRevenue, $logger);
    $subscriber->onOrderInsert($event);
  }

  /**
   * Tests onOrderInsert handles exceptions and logs logs.
   */
  public function testOnOrderInsertHandlesException(): void {
    $order = $this->createMock(OrderInterface::class);

    $exception = new \Exception('Test exception');
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('incrementFromOrder')
      ->with($order)
      ->willThrowException($exception);

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->never())
      ->method('incrementFromOrder');

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $event = $this->createMock(OrderEvent::class);
    $event->expects($this->once())
      ->method('getOrder')
      ->willReturn($order);

    $subscriber = new OrderMetricsSubscriber($orderTotal, $orderRevenue, $logger);
    $subscriber->onOrderInsert($event);
  }

  /**
   * Tests onOrderUpdate calls both metric services when original exists.
   */
  public function testOnOrderUpdate(): void {
    $originalOrder = $this->createMock(OrderInterface::class);

    $order = $this->createMock(OrderInterface::class);
    $order->method('getOriginal')->willReturn($originalOrder);

    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('updateFromOrders')
      ->with($originalOrder, $order);

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->once())
      ->method('updateFromOrders')
      ->with($originalOrder, $order);

    $logger = $this->createMock(LoggerInterface::class);

    $event = $this->createMock(OrderEvent::class);
    $event->expects($this->once())
      ->method('getOrder')
      ->willReturn($order);

    $subscriber = new OrderMetricsSubscriber($orderTotal, $orderRevenue, $logger);
    $subscriber->onOrderUpdate($event);
  }

  /**
   * Tests onOrderUpdate does nothing when original is missing.
   */
  public function testOnOrderUpdateWithoutOriginal(): void {
    $order = $this->createMock(OrderInterface::class);
    // No $order->original set.
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->never())
      ->method('updateFromOrders');

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->never())
      ->method('updateFromOrders');

    $logger = $this->createMock(LoggerInterface::class);

    $event = $this->createMock(OrderEvent::class);
    $event->expects($this->once())
      ->method('getOrder')
      ->willReturn($order);

    $subscriber = new OrderMetricsSubscriber($orderTotal, $orderRevenue, $logger);
    $subscriber->onOrderUpdate($event);
  }

  /**
   * Tests onOrderUpdate handles exceptions and logs logs.
   */
  public function testOnOrderUpdateHandlesException(): void {
    $originalOrder = $this->createMock(OrderInterface::class);

    $order = $this->createMock(OrderInterface::class);
    $order->method('getOriginal')->willReturn($originalOrder);

    $exception = new \Exception('Test exception');
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('updateFromOrders')
      ->willThrowException($exception);

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->never())
      ->method('updateFromOrders');

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $event = $this->createMock(OrderEvent::class);
    $event->expects($this->once())
      ->method('getOrder')
      ->willReturn($order);

    $subscriber = new OrderMetricsSubscriber($orderTotal, $orderRevenue, $logger);
    $subscriber->onOrderUpdate($event);
  }

  /**
   * Tests onOrderDelete calls both metric services.
   */
  public function testOnOrderDelete(): void {
    $order = $this->createMock(OrderInterface::class);

    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('decrementFromOrder')
      ->with($order);

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->once())
      ->method('decrementFromOrder')
      ->with($order);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->never())->method('log');

    $event = $this->createMock(OrderEvent::class);
    $event->expects($this->once())
      ->method('getOrder')
      ->willReturn($order);

    $subscriber = new OrderMetricsSubscriber($orderTotal, $orderRevenue, $logger);
    $subscriber->onOrderDelete($event);
  }

  /**
   * Tests onOrderDelete handles exceptions and logs logs.
   */
  public function testOnOrderDeleteHandlesException(): void {
    $order = $this->createMock(OrderInterface::class);

    $exception = new \Exception('Test exception');
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('decrementFromOrder')
      ->with($order)
      ->willThrowException($exception);

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->never())
      ->method('decrementFromOrder');

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $event = $this->createMock(OrderEvent::class);
    $event->expects($this->once())
      ->method('getOrder')
      ->willReturn($order);

    $subscriber = new OrderMetricsSubscriber($orderTotal, $orderRevenue, $logger);
    $subscriber->onOrderDelete($event);
  }

}
