<?php

declare(strict_types=1);

namespace Drupal\Tests\prometheus_metrics_commerce\Unit;

use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\commerce_product\Event\ProductEvent;
use Drupal\commerce_product\Event\ProductEvents;
use Drupal\prometheus_metrics_commerce\EventSubscriber\ProductMetricsSubscriber;
use Drupal\prometheus_metrics_commerce\Metrics\Product;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;

/**
 * Tests the ProductMetricsSubscriber event subscriber.
 */
#[Group("prometheus_metrics_commerce")]
class ProductMetricsSubscriberTest extends UnitTestCase {

  /**
   * Tests getSubscribedEvents returns correct event mapping.
   */
  public function testGetSubscribedEvents(): void {
    $events = ProductMetricsSubscriber::getSubscribedEvents();

    $this->assertArrayHasKey(ProductEvents::PRODUCT_INSERT, $events);
    $this->assertEquals(['onProductInsert'], $events[ProductEvents::PRODUCT_INSERT]);

    $this->assertArrayHasKey(ProductEvents::PRODUCT_UPDATE, $events);
    $this->assertEquals(['onProductUpdate'], $events[ProductEvents::PRODUCT_UPDATE]);

    $this->assertArrayHasKey(ProductEvents::PRODUCT_DELETE, $events);
    $this->assertEquals(['onProductDelete'], $events[ProductEvents::PRODUCT_DELETE]);
  }

  /**
   * Tests onProductInsert calls incrementFromProduct.
   */
  public function testOnProductInsert(): void {
    $product = $this->createMock(ProductInterface::class);

    $productMetric = $this->createMock(Product::class);
    $productMetric->expects($this->once())
      ->method('incrementFromProduct')
      ->with($product);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->never())->method('log');

    $event = $this->createMock(ProductEvent::class);
    $event->expects($this->once())
      ->method('getProduct')
      ->willReturn($product);

    $subscriber = new ProductMetricsSubscriber($productMetric, $logger);
    $subscriber->onProductInsert($event);
  }

  /**
   * Tests onProductInsert handles exceptions and logs logs.
   */
  public function testOnProductInsertHandlesException(): void {
    $product = $this->createMock(ProductInterface::class);

    $exception = new \Exception('Test exception');
    $productMetric = $this->createMock(Product::class);
    $productMetric->expects($this->once())
      ->method('incrementFromProduct')
      ->with($product)
      ->willThrowException($exception);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $event = $this->createMock(ProductEvent::class);
    $event->expects($this->once())
      ->method('getProduct')
      ->willReturn($product);

    $subscriber = new ProductMetricsSubscriber($productMetric, $logger);
    $subscriber->onProductInsert($event);
  }

  /**
   * Tests onProductUpdate calls updateFromProducts when original exists.
   */
  public function testOnProductUpdate(): void {
    $originalProduct = $this->createMock(ProductInterface::class);

    $product = $this->createMock(ProductInterface::class);
    $product->method('getOriginal')->willReturn($originalProduct);

    $productMetric = $this->createMock(Product::class);
    $productMetric->expects($this->once())
      ->method('updateFromProducts')
      ->with($originalProduct, $product);

    $logger = $this->createMock(LoggerInterface::class);

    $event = $this->createMock(ProductEvent::class);
    $event->expects($this->once())
      ->method('getProduct')
      ->willReturn($product);

    $subscriber = new ProductMetricsSubscriber($productMetric, $logger);
    $subscriber->onProductUpdate($event);
  }

  /**
   * Tests onProductUpdate does nothing when original is missing.
   */
  public function testOnProductUpdateWithoutOriginal(): void {
    $product = $this->createMock(ProductInterface::class);
    // No $product->original set.
    $productMetric = $this->createMock(Product::class);
    $productMetric->expects($this->never())
      ->method('updateFromProducts');

    $logger = $this->createMock(LoggerInterface::class);

    $event = $this->createMock(ProductEvent::class);
    $event->expects($this->once())
      ->method('getProduct')
      ->willReturn($product);

    $subscriber = new ProductMetricsSubscriber($productMetric, $logger);
    $subscriber->onProductUpdate($event);
  }

  /**
   * Tests onProductUpdate handles exceptions and logs logs.
   */
  public function testOnProductUpdateHandlesException(): void {
    $originalProduct = $this->createMock(ProductInterface::class);

    $product = $this->createMock(ProductInterface::class);
    $product->method('getOriginal')->willReturn($originalProduct);

    $exception = new \Exception('Test exception');
    $productMetric = $this->createMock(Product::class);
    $productMetric->expects($this->once())
      ->method('updateFromProducts')
      ->willThrowException($exception);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $event = $this->createMock(ProductEvent::class);
    $event->expects($this->once())
      ->method('getProduct')
      ->willReturn($product);

    $subscriber = new ProductMetricsSubscriber($productMetric, $logger);
    $subscriber->onProductUpdate($event);
  }

  /**
   * Tests onProductDelete calls decrementFromProduct.
   */
  public function testOnProductDelete(): void {
    $product = $this->createMock(ProductInterface::class);

    $productMetric = $this->createMock(Product::class);
    $productMetric->expects($this->once())
      ->method('decrementFromProduct')
      ->with($product);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->never())->method('log');

    $event = $this->createMock(ProductEvent::class);
    $event->expects($this->once())
      ->method('getProduct')
      ->willReturn($product);

    $subscriber = new ProductMetricsSubscriber($productMetric, $logger);
    $subscriber->onProductDelete($event);
  }

  /**
   * Tests onProductDelete handles exceptions and logs logs.
   */
  public function testOnProductDeleteHandlesException(): void {
    $product = $this->createMock(ProductInterface::class);

    $exception = new \Exception('Test exception');
    $productMetric = $this->createMock(Product::class);
    $productMetric->expects($this->once())
      ->method('decrementFromProduct')
      ->with($product)
      ->willThrowException($exception);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $event = $this->createMock(ProductEvent::class);
    $event->expects($this->once())
      ->method('getProduct')
      ->willReturn($product);

    $subscriber = new ProductMetricsSubscriber($productMetric, $logger);
    $subscriber->onProductDelete($event);
  }

}
