<?php

declare(strict_types=1);

namespace Drupal\Tests\prometheus_metrics_commerce\Unit;

use Drupal\prometheus_metrics_commerce\Metrics\OrderRevenue;
use Drupal\prometheus_metrics_commerce\Metrics\OrderTotal;
use Drupal\prometheus_metrics_commerce\Metrics\Product;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Drupal\prometheus_metrics\Events\RebuildMetricsEvent;
use Drupal\prometheus_metrics_commerce\EventSubscriber\RebuildMetricsEventSubscriber;

/**
 * Tests the RebuildMetricsEventSubscriber event subscriber.
 */
#[Group("prometheus_metrics_commerce")]
class RebuildMetricsEventSubscriberTest extends UnitTestCase {

  /**
   * Tests onRebuildMetrics calls rebuild on all metric services.
   */
  public function testOnRebuildMetrics(): void {
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('rebuild');

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->once())
      ->method('rebuild');

    $product = $this->createMock(Product::class);
    $product->expects($this->once())
      ->method('rebuild');

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->never())
      ->method('log');

    $metrics = $this->createMock(PrometheusMetricsInterface::class);
    $hook = new RebuildMetricsEventSubscriber($orderTotal, $orderRevenue, $product, $logger);
    $hook->onRebuildMetrics(new RebuildMetricsEvent($metrics));
  }

  /**
   * Tests onRebuildMetrics handles exceptions.
   *
   * Continues with remaining metrics.
   */
  public function testCacheRebuildHandlesExceptionInOrderTotal(): void {
    $exception = new \Exception('Test exception');
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('rebuild')
      ->willThrowException($exception);

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->once())
      ->method('rebuild');

    $product = $this->createMock(Product::class);
    $product->expects($this->once())
      ->method('rebuild');

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $hook = new RebuildMetricsEventSubscriber($orderTotal, $orderRevenue, $product, $logger);
    $metrics = $this->createMock(PrometheusMetricsInterface::class);
    $hook->onRebuildMetrics(new RebuildMetricsEvent($metrics));
  }

  /**
   * Tests onRebuildMetrics handles exceptions in OrderRevenue metric.
   */
  public function testCacheRebuildHandlesExceptionInOrderRevenue(): void {
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('rebuild');

    $exception = new \Exception('Test exception');
    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->once())
      ->method('rebuild')
      ->willThrowException($exception);

    $product = $this->createMock(Product::class);
    $product->expects($this->once())
      ->method('rebuild');

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $hook = new RebuildMetricsEventSubscriber($orderTotal, $orderRevenue, $product, $logger);
    $metrics = $this->createMock(PrometheusMetricsInterface::class);
    $hook->onRebuildMetrics(new RebuildMetricsEvent($metrics));
  }

  /**
   * Tests onRebuildMetrics handles exceptions in Product metric.
   */
  public function testCacheRebuildHandlesExceptionInProduct(): void {
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('rebuild');

    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->once())
      ->method('rebuild');

    $exception = new \Exception('Test exception');
    $product = $this->createMock(Product::class);
    $product->expects($this->once())
      ->method('rebuild')
      ->willThrowException($exception);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('log');

    $hook = new RebuildMetricsEventSubscriber($orderTotal, $orderRevenue, $product, $logger);
    $metrics = $this->createMock(PrometheusMetricsInterface::class);
    $hook->onRebuildMetrics(new RebuildMetricsEvent($metrics));
  }

  /**
   * Tests onRebuildMetrics handles multiple exceptions.
   */
  public function testCacheRebuildHandlesMultipleExceptions(): void {
    $exception1 = new \Exception('Exception 1');
    $orderTotal = $this->createMock(OrderTotal::class);
    $orderTotal->expects($this->once())
      ->method('rebuild')
      ->willThrowException($exception1);

    $exception2 = new \Exception('Exception 2');
    $orderRevenue = $this->createMock(OrderRevenue::class);
    $orderRevenue->expects($this->once())
      ->method('rebuild')
      ->willThrowException($exception2);

    $exception3 = new \Exception('Exception 3');
    $product = $this->createMock(Product::class);
    $product->expects($this->once())
      ->method('rebuild')
      ->willThrowException($exception3);

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->exactly(3))
      ->method('log');

    $hook = new RebuildMetricsEventSubscriber($orderTotal, $orderRevenue, $product, $logger);
    $metrics = $this->createMock(PrometheusMetricsInterface::class);
    $hook->onRebuildMetrics(new RebuildMetricsEvent($metrics));
  }

}
