<?php

namespace Drupal\prometheus_metrics\EventSubscriber;

use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Psr\Log\LoggerInterface;
use Drupal\Component\Utility\Timer;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Counts all requests/ total requests and provides request timer metrics.
 *
 * @package Drupal\prometheus_metrics\EventSubscriber
 */
class PrometheusRequestSubscriber implements EventSubscriberInterface {

  /**
   * Used to retrieve route details.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  private $routeMatch;

  /**
   * The prometheus metrics service.
   *
   * @var \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface
   */
  private $prometheusMetrics;

  /**
   * Used to access Prometheus Metrics config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  private $config;

  /**
   * Namespace retrieved from config.
   *
   * @var string
   */
  private $namespace;

  /**
   * Used to log messages.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private $logger;

  /**
   * List of routes to exclude from metrics.
   *
   * @var string[]
   */
  private $excludeRoutes;

  /**
   * List of routes to include in metrics.
   *
   * @var string[]
   */
  private $includeRoutes;

  /**
   * PrometheusRequestSubscriber constructor.
   *
   * @param \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface $prometheusMetrics
   *   The Prometheus Metrics service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   Used to retrieve route details.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config service.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger service.
   */
  public function __construct(
    PrometheusMetricsInterface $prometheusMetrics,
    RouteMatchInterface $routeMatch,
    ConfigFactoryInterface $configFactory,
    #[Autowire(service: 'logger.channel.prometheus_metrics')]
    LoggerInterface $logger,
  ) {
    $this->routeMatch = $routeMatch;
    $this->prometheusMetrics = $prometheusMetrics;
    $this->config = $configFactory->get(PrometheusDefaults::CONFIGURATION_NAME);
    $this->namespace = $this->config->get('metrics_namespace') ? $this->config->get('metrics_namespace') : PrometheusDefaults::METRICS_NAMESPACE;
    $this->excludeRoutes = $this->config->get('metrics_exclude_routes') ? $this->config->get('metrics_exclude_routes') : [];
    $this->includeRoutes = $this->config->get('metrics_include_routes') ? $this->config->get('metrics_include_routes') : [];
    $this->logger = $logger;
  }

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => ['onRequest', 9999],
      KernelEvents::TERMINATE => ['onTerminate', 0],
    ];
  }

  /**
   * Start timer for the request.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The start of request event.
   */
  public function onRequest(RequestEvent $event) {
    Timer::start(PrometheusDefaults::TIMER_NAME);

    // Track PHP version info - we only need to set this once per process.
    $phpVersionGauge = $this->prometheusMetrics->getGauge(
          $this->namespace,
          'php_info',
          'PHP version information',
          ['version']
      );
    $phpVersionGauge->set(1, [PHP_VERSION]);
  }

  /**
   * Checks if route is in list of excludes routes.
   *
   * @param string $routeName
   *   The route name to exclude.
   *
   * @return bool
   *   Returns TRUE if the route is in the exclude list, FALSE otherwise.
   */
  private function excludeRoute(string $routeName) {
    if (!empty($this->excludeRoutes) && is_array($this->excludeRoutes)) {
      foreach ($this->excludeRoutes as $excludeRoute) {
        $regex = str_replace(['*', '/'], ['([\s\S]+)', '\\/'], $excludeRoute);
        if (preg_match("/^$regex$/", $routeName, $matches) === 1) {
          return TRUE;
        }
      }
    }
    return FALSE;
  }

  /**
   * Checks if route should be included in metrics.
   *
   * @param string $routeName
   *   The route name to check.
   *
   * @return bool
   *   Returns TRUE if the route should be included in metrics, FALSE otherwise.
   */
  private function shouldIncludeRoute(string $routeName) {
    // If no include routes specified, include all routes except excluded ones.
    if (empty($this->includeRoutes)) {
      return !$this->excludeRoute($routeName);
    }

    // Check if route matches any include patterns.
    foreach ($this->includeRoutes as $includeRoute) {
      $regex = str_replace(['*', '/'], ['([\s\S]+)', '\\/'], $includeRoute);
      if (preg_match("/^$regex$/", $routeName, $matches) === 1) {
        // Still check exclude rules.
        return !$this->excludeRoute($routeName);
      }
    }

    // Log non-included routes if configured to do so.
    if ($this->config->get('log_non_included_routes')) {
      $this->logger->info(
            'Route @route not included in specific metrics (but tracked in allRoutes)', [
              '@route' => $routeName,
            ]
        );
    }

    // Route is not in include list, but may still be tracked in allRoutes.
    return FALSE;
  }

  /**
   * Handles the terminate event.
   *
   * Stores a histogram timer and counter for the request
   * by method, route and status.
   *
   * @param \Symfony\Component\HttpKernel\Event\TerminateEvent $event
   *   The event.
   */
  public function onTerminate(TerminateEvent $event) {
    if (PHP_SAPI === "cli") {
      // In CLI context there is no request to trace.
      return;
    }

    Timer::stop(PrometheusDefaults::TIMER_NAME);
    $timeInMs = Timer::read(PrometheusDefaults::TIMER_NAME);
    $method = $event->getRequest()->getMethod();
    $routeName = $this->getRouteName($event->getRequest());

    // Get peak memory in MB.
    $peakMemoryMB = memory_get_peak_usage(TRUE) / 1024 / 1024;

    if (!$this->shouldIncludeRoute($routeName)) {
      // We want to track it in allRoutes except those specifically excluded.
      if ($this->excludeRoute($routeName)) {
        return;
      }
      // Skip individual route metrics but continue to allRoutes metrics.
    }
    else {
      if (empty($timeInMs)) {
        $this->logger->warning(
              'Timer not found for: @method @routeName',
              [
                '@method' => $method,
                '@routeName' => $routeName,
              ]
          );
      }
      $timeInS = $timeInMs / 1000;

      // Calls function to get/register counter based on namespace and name.
      $counter = $this->prometheusMetrics->getCounter(
            $this->namespace,
            'http_requests_total',
            'Total number of requests',
            [
              'method',
              'route',
              'status',
            ]
        );

      $counter->inc([$method, $routeName, $this->flattenResponseCode($event->getResponse())]);

      // Calls function to get/register histogram based on namespace and name.
      $histogram = $this->prometheusMetrics->getHistogram(
            $this->namespace,
            'http_requests_total',
            'Timer for all requests',
            [
              'method',
              'route',
              'status',
            ]
        );

      $histogram->observe($timeInS, [$method, $routeName, $this->flattenResponseCode($event->getResponse())]);

      // Track peak memory for specific route.
      $memoryGauge = $this->prometheusMetrics->getGauge(
            $this->namespace,
            'peak_memory_mb',
            'Peak memory usage in megabytes',
            [
              'method',
              'route',
              'status',
            ]
        );
      $memoryGauge->set($peakMemoryMB, [$method, $routeName, $this->flattenResponseCode($event->getResponse())]);
    }

    // Add consolidated metrics for all non-excluded routes.
    $timeInS = $timeInMs / 1000;
    $allRoutesCounter = $this->prometheusMetrics->getCounter(
          $this->namespace,
          'http_requests_total',
          'Total number of requests across all routes',
          [
            'method',
            'route',
            'status',
          ]
      );

    $allRoutesCounter->inc([$method, 'allRoutes', $this->flattenResponseCode($event->getResponse())]);

    $allRoutesHistogram = $this->prometheusMetrics->getHistogram(
          $this->namespace,
          'http_requests_total',
          'Timer for all requests across all routes',
          [
            'method',
            'route',
            'status',
          ]
      );

    $allRoutesHistogram->observe($timeInS, [$method, 'allRoutes', $this->flattenResponseCode($event->getResponse())]);

    // Track peak memory for allRoutes.
    $allRoutesMemoryGauge = $this->prometheusMetrics->getGauge(
          $this->namespace,
          'peak_memory_mb',
          'Peak memory usage in megabytes across all routes',
          [
            'method',
            'route',
            'status',
          ]
      );
    $allRoutesMemoryGauge->set($peakMemoryMB, [$method, 'allRoutes', $this->flattenResponseCode($event->getResponse())]);
  }

  /**
   * Extracts the route name from the request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return string|null
   *   returns null or the request uri.
   */
  private function getRouteName(Request $request): ?string {
    // Does routeMatch return a value.
    if (!empty($this->routeMatch->getRouteName())) {
      $routeName = $this->routeMatch->getRouteName();
      return str_replace(".", "_", $routeName);
    }
    else {
      return $request->getRequestUri();
    }
  }

  /**
   * Flattens Response Status codes into 4 strings 2xx - 5xx.
   *
   * @param \Symfony\Component\HttpFoundation\Response $response
   *   The response object.
   *
   * @return int|string
   *   Returns the status as a 'grouped' string or the actual status code.
   */
  private function flattenResponseCode(Response $response) {
    if ($response->getStatusCode() >= 200 && $response->getStatusCode() < 300) {
      return '2xx';
    }
    elseif ($response->getStatusCode() >= 300 && $response->getStatusCode() < 400) {
      return '3xx';
    }
    elseif ($response->getStatusCode() >= 400 && $response->getStatusCode() < 500) {
      return '4xx';
    }
    elseif ($response->getStatusCode() >= 500) {
      return '5xx';
    }
    return $response->getStatusCode();
  }

}
