<?php

namespace Drupal\prometheus_metrics\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Rebuilds metrics after a registry wipe.
 *
 * Currently only entity create counters can be rebuilt because update/delete
 * totals are not persisted anywhere.
 */
class RebuildMetricsSubscriber extends AbstractRebuildMetricsSubscriber {

  /**
   * Constructs the subscriber.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $bundleInfo,
    protected ConfigFactoryInterface $configFactory,
    #[Autowire(service: 'logger.channel.prometheus_metrics')]
    protected LoggerInterface $logger,
  ) {
  }

  /**
   * Rebuilds the Entity CRUD metrics where possible.
   */
  public function rebuildMetrics(PrometheusMetricsInterface $metrics): void {
    $config = $this->configFactory->get(PrometheusDefaults::CONFIGURATION_NAME);
    $namespace = $config->get('metrics_namespace') ?: PrometheusDefaults::METRICS_NAMESPACE;

    $createCounter = $metrics->getCounter(
          $namespace,
          'entity_create',
          'Counts when entity created',
          ['create']
      );

    foreach ($this->entityTypeManager->getDefinitions() as $entityTypeId => $entityType) {
      if (!$entityType instanceof ContentEntityTypeInterface) {
        continue;
      }

      $bundleKey = $entityType->getKey('bundle');
      try {
        $bundleIds = $bundleKey
                ? array_keys($this->bundleInfo->getBundleInfo($entityTypeId))
                : [$entityTypeId];
      }
      catch (\Exception $e) {
        $this->logger->warning(
              'Unable to determine bundles for @entity_type: @message',
              [
                '@entity_type' => $entityTypeId,
                '@message' => $e->getMessage(),
              ]
          );
        continue;
      }

      if (empty($bundleIds)) {
        $this->logger->warning(
              'No bundles found while rebuilding metrics for @entity_type.',
              ['@entity_type' => $entityTypeId]
          );
        continue;
      }

      foreach ($bundleIds as $bundleId) {
        $count = $this->countEntities($entityTypeId, $bundleKey, $bundleId);
        if ($count === 0) {
          continue;
        }

        if (method_exists($createCounter, 'incBy')) {
          $createCounter->incBy($count, [$bundleId]);
        }
        else {
          for ($i = 0; $i < $count; $i++) {
            $createCounter->inc([$bundleId]);
          }
        }
      }
    }
  }

  /**
   * Counts entities for an entity type/bundle combination.
   */
  protected function countEntities(string $entityTypeId, ?string $bundleKey, ?string $bundleId = NULL): int {
    try {
      $query = $this->entityTypeManager->getStorage($entityTypeId)
        ->getQuery()
        ->accessCheck(FALSE);

      if ($bundleKey && $bundleId) {
        $query->condition($bundleKey, $bundleId);
      }

      return (int) $query->count()->execute();
    }
    catch (\Exception $e) {
      $this->logger->warning(
            'Unable to rebuild metrics for @entity_type (@bundle): @message',
            [
              '@entity_type' => $entityTypeId,
              '@bundle' => $bundleId ?? $entityTypeId,
              '@message' => $e->getMessage(),
            ]
        );
    }

    return 0;
  }

}
