<?php

namespace Drupal\prometheus_metrics\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Drupal\prometheus_metrics\Events\RebuildMetricsEvent;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Allows user to cancel request to wipe all metrics.
 */
class ConfirmResetAndRebuildForm extends ConfirmFormBase {

  /**
   * The prometheus metrics service.
   *
   * @var \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface
   */
  private PrometheusMetricsInterface $prometheusMetrics;

  /**
   * Logger channel for rebuild operations.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private LoggerInterface $logger;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Event dispatcher used to trigger rebuild subscribers.
   *
   * @var \Symfony\Contracts\EventDispatcher\EventDispatcherInterface
   */
  private EventDispatcherInterface $eventDispatcher;

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface $prometheusMetrics
   *   The prometheus metrics service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    PrometheusMetricsInterface $prometheusMetrics,
    LoggerInterface $logger,
    EventDispatcherInterface $eventDispatcher,
    MessengerInterface $messenger,
  ) {
    $this->prometheusMetrics = $prometheusMetrics;
    $this->logger = $logger;
    $this->eventDispatcher = $eventDispatcher;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('prometheus_metrics.promphp_bridge'),
          $container->get('logger.channel.prometheus_metrics'),
          $container->get('event_dispatcher'),
          $container->get('messenger'),
      );
  }

  /**
   * {@inheritDoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to wipe and rebuild all metrics generated so far?');
  }

  /**
   * {@inheritDoc}
   */
  public function getCancelUrl() {
    return new Url('prometheus_metrics.configuration_form');
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return "prometheus_metrics.rebuild";
  }

  /**
   * Handles submission, attempting to wipe and rebuild from storage.
   *
   * @param array $form
   *   The confirm reset and rebuild form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   State of confirm reset and rebuild form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Delete metrics.
    if ($this->prometheusMetrics->wipeRegistry()) {
      // Dispatch event to rebuild metrics.
      $event = new RebuildMetricsEvent($this->prometheusMetrics);
      $this->eventDispatcher->dispatch($event);

      $this->logger->info('All metrics successfully wiped and rebuilt.');

      $this->messenger->addMessage(
        $this->t('All metrics successfully wiped and rebuilt.'),
        MessengerInterface::TYPE_STATUS,
      );
    }
    else {
      $this->messenger->addMessage(
        $this->t('Failed to wipe and rebuild metrics.'),
        MessengerInterface::TYPE_ERROR,
      );
    }
    // Redirect back to manage form.
    $form_state->setRedirect('prometheus_metrics.configuration_form');

  }

}
