<?php

namespace Drupal\prometheus_metrics\Form;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteBuilderInterface;
use Drupal\Core\Url;
use Drupal\prometheus_metrics\EventSubscriber\PrometheusDefaults;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Definition of prometheus_metrics route configuration form.
 */
class MetricsConfigurationForm extends ConfigFormBase {
  /**
   * Used to rebuild route if route name provided.
   *
   * @var \Drupal\Core\Routing\RouteBuilderInterface
   */
  private $routeBuilder;

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Routing\RouteBuilderInterface $routeBuilder
   *   Route builder service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface|null $typed_config_manager
   *   The typed config manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RouteBuilderInterface $routeBuilder,
    ?TypedConfigManagerInterface $typed_config_manager = NULL,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->routeBuilder = $routeBuilder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('config.factory'),
          $container->get('router.builder'),
          $container->has('config.typed') ? $container->get('config.typed') : NULL
      );
  }

  /**
   * {@inheritDoc}
   */
  protected function getEditableConfigNames() {
    return [PrometheusDefaults::CONFIGURATION_NAME];
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return 'prometheus_metrics_route_configuration_form';
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(PrometheusDefaults::CONFIGURATION_NAME);
    $form['metrics_route_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Metrics path'),
      '#description' => $this->t('Path to be used by Prometheus to collect metrics. Include preceding slash e.g. /manage/prometheus'),
      '#default_value' => $config->get('metrics_route_path') ?? '/prometheus/metrics',
    ];

    $form['metrics_namespace'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Metrics namespace'),
      '#description' => $this->t('Namespace used for prometheus metrics. Should be unique to this service'),
      '#default_value' => $config->get('metrics_namespace') ?? 'drupal',
    ];

    // @todo make routes collected configurable, and review prom_php to see if possible to reduce memory usage.
    $form['metrics_route_memory'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Metrics Memory Ini Set'),
      '#description' => $this->t('(experimental) This module can collect a lot of metrics. Rendering metrics out can be expensive. Specify a memory limit for this route e.g. 256M. If not specified default ini values will be used.'),
      '#default_value' => $config->get('metrics_route_memory') ?? '',
    ];

    $form['metrics_include_routes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Include routes'),
      '#description' => $this->t('Line separated list of routes to include. If empty, all routes will be included. Wildcards may be used e.g. entity.node.*'),
      '#default_value' => is_array($config->get('metrics_include_routes')) ? implode("\r\n", $config->get('metrics_include_routes')) : '',
    ];

    $form['metrics_exclude_routes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exclude routes'),
      '#description' => $this->t('Line separated list of routes to exclude. Wildcards may be used e.g. prometheus_*'),
      '#default_value' => is_array($config->get('metrics_exclude_routes')) ? implode("\r\n", $config->get('metrics_exclude_routes')) : '',
    ];

    $form['log_non_included_routes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log non-included routes'),
      '#description' => $this->t('If checked, routes that are not specifically included (but still tracked in allRoutes) will be logged. This can help identify routes that might need specific monitoring.'),
      '#default_value' => $config->get('log_non_included_routes') ?? FALSE,
    ];

    $form['rebuild_metrics'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form-item']],
      'link' => [
        '#type' => 'link',
        '#title' => $this->t('Wipe and rebuild all metrics.'),
        '#url' => Url::fromRoute('prometheus_metrics.wipe_and_rebuild'),
      ],
      'description' => [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#value' => $this->t('Metrics will be wiped and rebuilt where possible and implemented. Not all metrics will be rebuilt, use with caution.'),
        '#attributes' => ['class' => ['description']],
      ],
    ];

    $form['require_auth'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require Authentication'),
      '#default_value' => $config->get('require_auth') ?? TRUE,
      '#description' => $this->t('If checked, authentication will be required to access the metrics endpoint. Add a user with the "access prometheus metrics" permission to the list of users who can access the metrics endpoint.'),
    ];

    $form['rebuild_metrics_on_cache_rebuild'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Rebuild metrics on cache rebuild'),
      '#default_value' => $config->get('rebuild_metrics_on_cache_rebuild') ?? FALSE,
      '#description' => $this->t('If checked, metrics will be rebuilt when the cache is rebuilt. <strong>It is highly recommended to enable this if metrics are stored in Redis and Redis is not configured to persist data</strong>. This is common for many Drupal hosting platforms, including Pantheon. This can be useful to ensure that metrics are up to date when the cache is rebuilt.'),
    ];

    $form['reset_metrics'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form-item']],
      'link' => [
        '#type' => 'link',
        '#title' => $this->t('Reset all metrics.'),
        '#url' => Url::fromRoute('prometheus_metrics.wipe'),
      ],
      'description' => [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#value' => $this->t('All metrics will be lost.'),
        '#attributes' => ['class' => ['description']],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritDoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $path = $form_state->getValue('metrics_route_path');

    if (!UrlHelper::isValid($path)) {
      $form_state->setErrorByName('metrics_route_path', $this->t('Path provided is invalid'));
    }
    // @todo Change the autogenerated stub
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $newRoute = $form_state->getValue('metrics_route_path');
    $oldRoute = $this->config(PrometheusDefaults::CONFIGURATION_NAME)
      ->get('metrics_route_path');

    $includeRoutes = array_filter(
        array_unique(
            explode("\r\n", $form_state->getValue('metrics_include_routes'))
        ),
        function ($item) {
            return !empty($item);
        }
    );

    $excludeRoutes = array_filter(
      array_unique(
          explode("\r\n", $form_state->getValue('metrics_exclude_routes'))
      ),
      function ($item) {
          return !empty($item);
      }
    );

    $this->config(PrometheusDefaults::CONFIGURATION_NAME)
      ->set('metrics_namespace', $form_state->getValue('metrics_namespace'))
      ->set('require_auth', $form_state->getValue('require_auth'))
      ->set('metrics_route_memory', $form_state->getValue('metrics_route_memory'))
      ->set('rebuild_metrics_on_cache_rebuild', $form_state->getValue('rebuild_metrics_on_cache_rebuild'))
      ->set('metrics_include_routes', $includeRoutes)
      ->set('metrics_exclude_routes', $excludeRoutes)
      ->set('metrics_route_path', $newRoute)
      ->set('log_non_included_routes', $form_state->getValue('log_non_included_routes'))
      ->save();

    if ($newRoute != $oldRoute) {
      $this->routeBuilder->rebuild();
    }

    parent::submitForm($form, $form_state);
  }

}
