<?php

declare(strict_types=1);

namespace Drupal\prometheus_metrics\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\prometheus_metrics\Events\RebuildMetricsEvent;

/**
 * Rebuild metrics when cache is rebuilt.
 *
 * This allows for subscribers to rebuild the metrics they care about.
 */
class CacheRebuildMetrics {

  /**
   * Constructs a new CacheRebuildMetrics instance.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher service.
   * @param \Drupal\prometheus_metrics\Bridge\PrometheusMetricsInterface $prometheusMetrics
   *   The prometheus metrics service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(
    #[Autowire(service: 'logger.channel.prometheus_metrics')]
    protected LoggerInterface $logger,
    protected EventDispatcherInterface $eventDispatcher,
    protected PrometheusMetricsInterface $prometheusMetrics,
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * Implements hook_rebuild().
   */
  #[Hook('rebuild')]
  public function cacheRebuild(): void {
    // Config item says rebuild metrics.
    if ($this->configFactory->get('prometheus_metrics.configuration')->get('rebuild_metrics_on_cache_rebuild')) {
      $this->eventDispatcher->dispatch(new RebuildMetricsEvent($this->prometheusMetrics));
    }
  }

}
