import { NodeSpec, DOMOutputSpec } from 'prosemirror-model';
import { Plugin } from 'prosemirror-state';
import { MenuItem } from 'prosemirror-menu';
import { Schema } from 'prosemirror-model';
import { ComponentDefinition } from '../wrappers/config';
import { ApplicationWrapper } from '../wrappers/interfaces';
import { canInsert } from '../import/menu';

export interface BaseNodeConfig {
  name: string;
  displayName: string;
  tag: string;
  inline?: boolean;
  defining?: boolean;
  selectable?: boolean;
  code?: boolean;
  marks?: string;
  attrs?: Record<string, any>;
  parseDOM?: Array<{tag: string}>;
  content?: string;
  group?: string;
  nodeType: 'base';
}

export const baseNodeTemplate = (spec: BaseNodeConfig, appWrapper: ApplicationWrapper): ComponentDefinition => {
  // Create the DOM output spec
  const toDOM = (node?: any): DOMOutputSpec => {
    const tag = spec.tag;
    
    // Special handling for headings with level attribute
    if (tag.match(/^h\d$/) && node?.attrs?.level) {
      return [`h${node.attrs.level}`, 0];
    }
    
    // Self-closing tags
    if (tag === 'hr' || tag === 'br') {
      return [tag];
    }
    
    // Tags with content
    /*if (tag === 'pre') {
      return ['pre', ['code', 0]];
    }*/
    
    return [tag, 0];
  };

  // Build the node spec
  const nodeSpec: NodeSpec = {
    content: spec.content,
    group: spec.group,
    inline: spec.inline,
    defining: spec.defining,
    selectable: spec.selectable,
    code: spec.code,
    marks: spec.marks === '_' ? undefined : spec.marks,
    attrs: spec.attrs,
    parseDOM: spec.parseDOM,
    toDOM,
  };

  // Remove undefined properties
  Object.keys(nodeSpec).forEach(key => {
    if (nodeSpec[key as keyof NodeSpec] === undefined) {
      delete nodeSpec[key as keyof NodeSpec];
    }
  });

  // Create menu item
  const menuItem = (schema: Schema): MenuItem => {
    return new MenuItem({
      title: `Insert ${spec.displayName}`,
      label: spec.displayName,
      enable(state) { 
        // For inline nodes, check if we can insert at cursor
        if (spec.inline) {
          return state.selection.$from.parent.inlineContent;
        }
        // For block nodes, check if we can insert
        return canInsert(state, schema.nodes[spec.name]);
      },
      run(state, dispatch) {
        const node = schema.nodes[spec.name];
        if (!node) return;
        
        if (spec.inline) {
          // Insert inline node
          dispatch?.(state.tr.replaceSelectionWith(node.create()));
        } else {
          // Insert block node
          const { $from } = state.selection;
          const pos = $from.after();
          dispatch?.(state.tr.insert(pos, node.create()));
        }
      }
    });
  };

  // For base nodes, we don't need custom plugins
  const plugin = new Plugin({});

  return {
    name: spec.name,
    spec: nodeSpec,
    plugin,
    menuItem,
  };
};
