import { Node, NodeSpec } from 'prosemirror-model';
import { EditorView, NodeView } from 'prosemirror-view';

export abstract class BlockView implements NodeView {
  dom: HTMLElement;
  contentDOM?: HTMLElement;
  node: Node;
  view: EditorView;
  getPos: () => number | undefined;
  
  protected headerDOM: HTMLElement;
  protected headerTitleDOM: HTMLElement;
  protected bodyDOM: HTMLElement;
  protected menuDOM: HTMLElement;
  private menuButton: HTMLElement;
  protected floatingMenu: HTMLElement | null = null;
  private closeMenuHandler: (e: MouseEvent) => void = () => {};
  protected doOpenMenu: boolean = false;

  // Static method to extend node specs with updatedAt attribute
  static extendNodeSpec(spec: NodeSpec): NodeSpec {
    return {
      ...spec,
      attrs: {
        ...spec.attrs,
        updatedAt: { default: null }
      }
    };
  }

  constructor(node: Node, view: EditorView, getPos: () => number | undefined) {
    this.node = node;
    this.view = view;
    this.getPos = getPos;
    
    // Create main wrapper
    this.dom = document.createElement('div');
    this.dom.className = 'block-wrapper';
    
    // Create header section
    this.headerDOM = document.createElement('div');
    this.headerDOM.className = 'block-header';
    this.headerDOM.setAttribute('contenteditable', 'false');
    this.dom.appendChild(this.headerDOM);
    
    // Create header title container
    this.headerTitleDOM = document.createElement('div');
    this.headerTitleDOM.className = 'block-header-title';
    this.headerDOM.appendChild(this.headerTitleDOM);
    
    // Create menu in header
    this.menuDOM = document.createElement('div');
    this.menuDOM.className = 'block-menu';
    this.headerDOM.appendChild(this.menuDOM);
    
    // Create menu button
    this.menuButton = document.createElement('button');
    this.menuButton.className = 'block-menu-button';
    this.menuButton.innerText = '⋯';
    this.menuButton.setAttribute('title', 'Block options');
    this.menuDOM.appendChild(this.menuButton);
    
    // Create body section
    this.bodyDOM = document.createElement('div');
    this.bodyDOM.className = 'block-body';
    this.dom.appendChild(this.bodyDOM);

    this.setupMenu();
    this.updateHeader();
  }

  private setupMenu() {
    // Toggle menu on button click
    this.menuButton.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      this.toggleMenu();
    });

    // Close menu when clicking outside
    this.closeMenuHandler = (e: MouseEvent) => {
      if (this.floatingMenu && !this.floatingMenu.contains(e.target as HTMLElement) && !this.menuButton.contains(e.target as HTMLElement)) {
        this.closeMenu();
      }
    };
    document.addEventListener('click', this.closeMenuHandler);
  }

  private toggleMenu() {
    const pos = this.getPos();
    if (pos === undefined) return;

    if(this.doOpenMenu) return;

    this.doOpenMenu = !this.floatingMenu || this.floatingMenu.classList.contains('hidden');
    const tr = this.view.state.tr.setNodeMarkup(pos, undefined, {
      ...this.node.attrs,
      updatedAt: Date.now()
    });
    tr.setMeta("addToHistory", false);
    this.view.dispatch(tr);
  }

  private openMenu() {
    // Create floating menu if it doesn't exist
    if (!this.floatingMenu) {
      this.floatingMenu = document.createElement('div');
      this.floatingMenu.className = 'block-floating-menu';
      
      // Add remove option
      const removeButton = document.createElement('button'); 
      removeButton.className = 'block-menu-item';
      removeButton.textContent = 'Remove';
      removeButton.onclick = () => {
        const pos = this.getPos();
        if (pos !== undefined) {
          const tr = this.view.state.tr.delete(pos, pos + this.node.nodeSize);
          this.view.dispatch(tr);
        }
        this.closeMenu();
      };
      this.floatingMenu.appendChild(removeButton);

      // Add custom menu items
      this.addCustomMenuItems(this.floatingMenu);
      
      // Append to menu DOM since parent is already relatively positioned
      this.menuDOM.appendChild(this.floatingMenu);
    }

    this.floatingMenu.classList.remove('hidden');
    this.menuButton.classList.add('active');
  }

  protected closeMenu() {
    const pos = this.getPos();
    if (pos === undefined) return;

    this.doOpenMenu = false;
    const tr = this.view.state.tr.setNodeMarkup(pos, undefined, {
      ...this.node.attrs,
      updatedAt: Date.now()
    });
    tr.setMeta("addToHistory", false);
    this.view.dispatch(tr);
  }

  update(node: Node) {
    if (node.type !== this.node.type) return false;
    this.node = node;

    // Handle menu state based on local property
    if (this.doOpenMenu) {
      // Ensure menu is created and positioned
      if (!this.floatingMenu) {
        this.openMenu();
      } else {
        this.floatingMenu.classList.remove('hidden');
        this.menuButton.classList.add('active');
      }
      this.doOpenMenu = false;
    } else {
      if (this.floatingMenu) {
        this.floatingMenu.classList.add('hidden');
        this.menuButton.classList.remove('active');
      }
    }

    this.updateHeader();
    return true;
  }

  // Abstract method that must be implemented by child classes
  protected abstract updateHeader(): void;

  // Method to be overridden by child classes to add custom menu items
  protected addCustomMenuItems(menu: HTMLElement): void {
    // Default implementation does nothing
  }

  destroy() {
    // Clean up event listeners and floating menu
    document.removeEventListener('click', this.closeMenuHandler);
    this.menuButton.removeEventListener('click', this.toggleMenu);
    if (this.floatingMenu) {
      this.floatingMenu.remove();
      this.floatingMenu = null;
    }
  }
}
