import { NodeSpec, Node } from 'prosemirror-model';
import { Plugin } from 'prosemirror-state';
import { MenuItem } from 'prosemirror-menu';
import { canInsert } from '../import/menu';
import { Schema } from 'prosemirror-model';
import { EditorView } from 'prosemirror-view';
import { TextSelection } from 'prosemirror-state';
import { BlockView } from '../core/block-view';
import { ComponentDefinition } from '../wrappers/config';
import { ApplicationWrapper } from '../wrappers/interfaces';
import { SelectOption } from '../renderers/interfaces';

export interface LeafBlockConfig extends Partial<NodeSpec> {
  name: string;
  displayName: string;
  classes?: string[];
  editable?: boolean;
  variants?: Record<string, string>;
  createContentDOM?: (container: HTMLElement) => HTMLElement;
  disableFormatting?: boolean;
}

export interface LeafBlockDialogState {
  isOpen: boolean;
  type: 'edit' | null;
  data?: any;
}

// Helper function to create a properly initialized leaf block
export function createLeafBlockNode(schema: Schema, blockName: string, attrs?: { class?: string, index?: null|number }, textOnly: boolean = false) {
  if (textOnly) {
    return schema.nodes[blockName].create(attrs, schema.text(' '));
  }

  const paragraph = schema.nodes.paragraph.create(null, schema.text(' '));
  return schema.nodes[blockName].create(attrs, paragraph);
}

// Edit dialog
function openEditDialog(currentName: string, variants: Record<string, string> | undefined, currentVariant: string | null, callback: (name: string | null, variant: string | null) => void, appWrapper: ApplicationWrapper) {
  const renderer = appWrapper.getRendererFactory();
  
  // Create dialog content
  const dialogContentResult = renderer.elements.div({ className: 'edit-dialog-content' });
  const dialogContent = dialogContentResult.element;
  
  const inputResult = renderer.elements.input({
    type: 'text',
    value: currentName,
    placeholder: 'Enter name...',
    className: 'edit-dialog-input'
  });
  const input = inputResult.element;
  
  let variantSelect: HTMLSelectElement | null = null;
  if (variants) {
    const options: SelectOption[] = [
      { value: '', label: 'No variant', selected: !currentVariant }
    ];
    
    // Add variant options
    Object.entries(variants).forEach(([key, label]) => {
      options.push({
        value: key,
        label: label,
        selected: key === currentVariant
      });
    });
    
    const variantSelectResult = renderer.elements.select(options, { className: 'edit-dialog-variant' });
    variantSelect = variantSelectResult.element;
    dialogContent.appendChild(variantSelect);
  }
  
  const buttonContainerResult = renderer.elements.div({ className: 'edit-dialog-buttons' });
  const buttonContainer = buttonContainerResult.element;
  
  let closed = false;
  const closeCallback = () => {
    if (!closed) {
      closed = true;
      callback(null, null); // Call with null to indicate cancellation
    }
  };
  
  const saveButtonResult = renderer.elements.button('Save', {
    className: 'edit-dialog-save',
    listeners: {
      click: () => {
        closed = true;
        const selectedVariant = variantSelect ? variantSelect.value : null;
        callback(input.value, selectedVariant);
        dialog.close();
      }
    }
  });
  
  const cancelButtonResult = renderer.elements.button('Cancel', {
    className: 'edit-dialog-cancel',
    listeners: {
      click: () => {
        closeCallback();
        dialog.close();
      }
    }
  });
  
  buttonContainer.appendChild(saveButtonResult.element);
  buttonContainer.appendChild(cancelButtonResult.element);
  
  dialogContent.appendChild(input);
  dialogContent.appendChild(buttonContainer);
  
  // Create and open dialog using renderer
  const dialog = renderer.components.dialog({
    title: 'Edit Block',
    content: dialogContent,
    width: 'auto',
    height: 'auto',
    modal: true
  });
  
  input.focus();
  input.select();
  
  dialog.open();
}

export function createLeafBlock(config: LeafBlockConfig, appWrapper: ApplicationWrapper): ComponentDefinition {
  // Custom node view to handle leaf block rendering
  class LeafBlockView extends BlockView {
    private dialogOpen: boolean = false;
    protected doOpenEditModal: boolean = false;
    private appWrapper: ApplicationWrapper;

    constructor(node: Node, view: EditorView, getPos: () => number | undefined) {
      super(node, view, getPos);
      this.appWrapper = appWrapper;
      this.dom.className = `block-wrapper leaf-block ${config.name} ${node.attrs.class}`;
      
      if (config.editable) {
        this.setupEditing();
      }
    
      // Create content wrapper inside body
      if (config.createContentDOM) {
        this.contentDOM = config.createContentDOM(this.bodyDOM);
      } else {
        this.contentDOM = document.createElement('div');
        this.contentDOM.className = 'block-content';
        this.bodyDOM.appendChild(this.contentDOM);
      }
    }

    private setupEditing() {
      this.headerTitleDOM.classList.add('editable');
      this.headerTitleDOM.addEventListener('dblclick', () => this.openEditDialog());
    }

    protected addCustomMenuItems(menu: HTMLElement) {
      if (config.editable) {
        const editButton = document.createElement('button');
        editButton.className = 'block-menu-item';
        editButton.textContent = 'Edit';
        editButton.onclick = (e) => {
          e.preventDefault();
          e.stopPropagation();

          this.openEditDialog();
          this.closeMenu();
        };
        menu.insertBefore(editButton, menu.firstChild);
      }
    }

    private openEditDialog() {
      const pos = this.getPos();
      if (pos === undefined) return;

      this.doOpenEditModal = true;
      const tr = this.view.state.tr.setNodeMarkup(pos, undefined, {
        ...this.node.attrs,
        updatedAt: Date.now()
      });
      tr.setMeta("addToHistory", false);
      this.view.dispatch(tr);
    }

    protected updateHeader(): void {
      this.headerTitleDOM.textContent = '';
      
      if (this.node.attrs.index !== null && this.node.attrs.index !== undefined) {
        const counter = document.createElement('span');
        counter.className = 'block-index-counter';
        counter.textContent = `#${this.node.attrs.index}`;
        this.headerTitleDOM.appendChild(counter);
      }

      if (this.node.attrs.name) {
        const nameElement = document.createElement('strong');
        nameElement.textContent = this.node.attrs.name;
        this.headerTitleDOM.appendChild(nameElement);
      } else {
        this.headerTitleDOM.appendChild(document.createTextNode(config.displayName));
      }

      // Add variant badge if present
      if (this.node.attrs.variant && config.variants) {
        const variantLabel = config.variants[this.node.attrs.variant];
        if (variantLabel) {
          const variantBadge = document.createElement('span');
          variantBadge.className = 'block-variant-badge';
          variantBadge.textContent = variantLabel;
          this.headerTitleDOM.appendChild(variantBadge);
        }
      }
    }

    update(node: Node) {
      if (node.type !== this.node.type) return false;
      this.node = node;
      this.dom.className = `block-wrapper leaf-block ${config.name} ${node.attrs.class}`;
      
      // Handle dialogs
      if (this.doOpenEditModal) {
        this.doOpenEditModal = false;
        openEditDialog(node.attrs.name || '', config.variants, node.attrs.variant, (newName, newVariant) => {
          const pos = this.getPos();
          if (pos === undefined) return;

          const tr = this.view.state.tr.setNodeMarkup(pos, undefined, {
            ...this.node.attrs,
            name: newName === null ? this.node.attrs.name : newName,
            variant: newVariant === null ? this.node.attrs.variant : newVariant,
            updatedAt: Date.now()
          });
          tr.setMeta("addToHistory", false);
          this.view.dispatch(tr);
        }, this.appWrapper);
      }
      
      // Call parent's update method to handle menu state
      return super.update(node);
    }

    destroy() {
      this.dialogOpen = false;
      super.destroy();
    }
  }

  const spec = BlockView.extendNodeSpec({
    attrs: {
      class: { default: config.classes?.join(' ') || '' },
      name: { default: '' },
      index: { default: null },
      variant: { default: null },
      editDialog: { 
        default: { isOpen: false, type: null } as LeafBlockDialogState
      }
    },
    content: config.content || "leaf+",
    group: config.group || "block leaf_block",
    draggable: config.draggable ?? true,
    //isolating: config.isolating ?? true,
    toDOM: node => {
      const attrs = { 
        class: `${config.name} ${node.attrs.class}`,
        'data-name': node.attrs.name || '',
        'data-variant': node.attrs.variant || ''
      };
      return ["div", attrs, 0];
    },
    parseDOM: [{
      tag: `div.${config.name}`,
      getAttrs: dom => {
        const element = dom as HTMLElement;
        return { 
          class: element.className.replace(config.name, '').trim(),
          name: element.getAttribute('data-name') || '',
          variant: element.getAttribute('data-variant') || null,
          editDialog: { isOpen: false, type: null }
        };
      }
    }],
    ...config
  });

  const plugin = new Plugin({
    props: {
      nodeViews: {
        [config.name]: (node, view, getPos) => new LeafBlockView(node, view, getPos)
      },
      // Remove formatting when pasting into blocks with disableFormatting
      handlePaste: config.disableFormatting ? (view, event) => {
        const { state } = view;
        const { selection } = state;
        const node = selection.$anchor.node();

        if (node.type.name === config.name) {
          const text = event.clipboardData?.getData('text/plain');
          if (text) {
            event.preventDefault();
            const tr = state.tr.insertText(text);
            view.dispatch(tr);
            return true;
          }
        }
        return false;
      } : undefined
    }
  });

  const menuItem = (schema: Schema) => new MenuItem({
    title: `Insert ${config.displayName}`,
    label: config.displayName,
    enable(state) { 
      return canInsert(state, schema.nodes[config.name]);
    },
    run(state, dispatch) {
      const node = createLeafBlockNode(schema, config.name, undefined, config.content === 'text*');
      const tr = state.tr.replaceSelectionWith(node);
      const pos = tr.selection.from;
      tr.setSelection(TextSelection.create(tr.doc, pos));
      dispatch(tr);
    }
  });

  return { spec, plugin, menuItem, name: config.name };
} 
