export type FieldOptions = {
  /// The starting value for the field.
  value?: any

  /// The label for the field.
  label: string

  /// Whether the field is required.
  required?: boolean

  /// A function to validate the given value. Should return an
  /// error message if it is not valid.
  validate?: (value: any) => string | null

  /// A cleanup function for field values.
  clean?: (value: any) => any

  /// The ApplicationWrapper for DOM operations.
  app: import('../wrappers/interfaces').ApplicationWrapper;
}

/// The type of field that `openPrompt` expects to be passed to it.
export abstract class Field<Options extends FieldOptions=FieldOptions> {
  /// Create a field with the given options. Options support by all
  /// field types are:
  constructor(
    /// @internal
    readonly options: Options
  ) {}

  /// Render the field to the DOM. Should be implemented by all subclasses.
  abstract render(): HTMLElement

  onSubmit?: () => void;

  /// Read the field's value from its DOM node.
  read(dom: HTMLElement) { return (dom as any).value }

  /// A field-type-specific validation function.
  validateType(value: any): string | null { return null }

  /// @internal
  validate(value: any): string | null {
    if (!value && this.options.required)
      return "Required field"
    return this.validateType(value) || (this.options.validate ? this.options.validate(value) : null)
  }

  clean(value: any): any {
    return this.options.clean ? this.options.clean(value) : value
  }
}

/// A field class for single-line text fields.
export class TextField extends Field {
  render() {
    const renderer = this.options.app.getRendererFactory();
    const result = renderer.elements.input({
      type: "text",
      placeholder: this.options.label,
      value: this.options.value || "",
      autocomplete: "off"
    });
    return result.element;
  }
}

/// A field class for dropdown fields based on a plain `<select>`
/// tag. Expects an option `options`, which should be an array of
/// `{value: string, label: string}` objects, or a function taking a
/// `ProseMirror` instance and returning such an array.
export class SelectField extends Field {
  render() {
    const renderer = this.options.app.getRendererFactory();
    const options = (this.options as any).options as {value: string, label: string}[];
    
    const result = renderer.elements.select(
      options.map(o => ({
        value: o.value,
        label: o.label,
        selected: o.value === this.options.value
      }))
    );
    
    return result.element;
  }
}
