import { Node, NodeSpec } from "prosemirror-model";
import { EditorView } from "prosemirror-view";
import { BlockView } from "../core/block-view";
import { ApplicationWrapper } from "../wrappers/interfaces";

// Cache to store media previews by UUID
const mediaPreviewCache = new Map<string, { preview: string, label: string | null }>();

export class MediaView extends BlockView {
  private preview: string | null = null;
  private label: string | null = null;
  customDOM: HTMLElement;
  private appWrapper: ApplicationWrapper;

  setPreview(preview: string, label?: string|null) {
    this.preview = preview;
    this.label = label || null;

    const pos = this.getPos();
    if (pos === undefined) return;

    const tr = this.view.state.tr.setNodeMarkup(pos, undefined, {
      ...this.node.attrs,
      updatedAt: Date.now()
    });
    tr.setMeta("addToHistory", false);
    this.view.dispatch(tr);
  }

  constructor(node: Node, view: EditorView, getPos: () => any, appWrapper: ApplicationWrapper) {
    super(node, view, getPos);
    this.appWrapper = appWrapper;
    this.dom.setAttribute("data-entity-type", node.attrs["data-entity-type"]);
    this.dom.setAttribute("data-entity-uuid", node.attrs["data-entity-uuid"]);
    this.dom.classList.add('media-content');

    const uuid = node.attrs["data-entity-uuid"];

    // Create content wrapper inside body
    const renderer = this.appWrapper.getRendererFactory();
    const customDOMResult = renderer.elements.div({ className: 'block-content' });
    this.customDOM = customDOMResult.element;
    this.bodyDOM.appendChild(this.customDOM);

    // Check cache first
    const cachedPreview = mediaPreviewCache.get(uuid);
    if (cachedPreview) {
      this.preview = cachedPreview.preview;
      this.label = cachedPreview.label;
      this.update(node);
      return;
    }

    this.customDOM.innerText = "Loading...";

    // Use the application wrapper to get the media preview
    this.appWrapper.getMediaPreview(uuid)
      .then(({ preview, label }) => {
        if (preview) {
          // Store in cache
          mediaPreviewCache.set(uuid, { preview, label });
          this.preview = preview;
          this.label = label;
          this.update(node);
        }
      })
      .catch(error => {
        console.error("Failed to load media preview:", error);
        this.customDOM.innerText = "Failed to load media preview";
      });
  }

  update(node: Node) {
    if (node.type !== this.node.type) return false;
    this.node = node;

    // Update the preview if we have one
    if (this.preview) {
      const tempDiv = document.createElement('div');
      // Note: this.preview comes from trusted backend and is already sanitized
      tempDiv.innerHTML = this.preview;
      const image = tempDiv.querySelector("img");

      // Clear existing content
      while (this.customDOM.firstChild) {
        this.customDOM.removeChild(this.customDOM.firstChild);
      }

      // If we get an image, only use the image as preview
      if (image) {
        this.customDOM.appendChild(image);
      } else {
        // Note: this.preview comes from trusted backend and is already sanitized
        this.customDOM.innerHTML = this.preview;
      }

      if (this.label) {
        this.customDOM.setAttribute("title", this.label);
      }
    }

    // Call parent's update method to handle menu state
    return super.update(node);
  }

  protected updateHeader(): void {
    // Update header content based on node attributes
    const label = this.label || 'Media';
    this.headerTitleDOM.textContent = label;
  }
}

export const mediaNodeSpec: NodeSpec = BlockView.extendNodeSpec({
  attrs: {"data-entity-type": {default: "media"}, "data-entity-uuid": {default: ""}},
  inline: true,
  group: "inline",
  draggable: true,

  toDOM: node => ["drupal-media", {"data-entity-type": node.attrs["data-entity-type"],
                  "data-entity-uuid": node.attrs["data-entity-uuid"]}],
  // When parsing, such an image, if its type matches one of the known
  // types, is converted to a dino node.
  parseDOM: [{
    tag: "img[media-uuid]",
    getAttrs: dom => {
      let uuid = dom.getAttribute("media-uuid")
      return uuid ? {
        "data-entity-type": dom.getAttribute("data-entity-type") ?? "media",
        "data-entity-uuid": uuid,
      } : false
    }
  }, {
    tag: "drupal-media",
    getAttrs: dom => {
      let uuid = dom.getAttribute("data-entity-uuid")
      return uuid ? {
        "data-entity-type": dom.getAttribute("data-entity-type") ?? "media",
        "data-entity-uuid": uuid,
      } : false
    }
  }]
});
