import { Plugin } from "prosemirror-state";
import { EditorView } from "prosemirror-view";
import { addColumnAfter, addColumnBefore, deleteColumn, addRowAfter, addRowBefore, deleteRow, selectedRect, TableRect } from "prosemirror-tables";
import { ApplicationWrapper } from "../wrappers/interfaces";

interface TableTooltipOptions {
  appWrapper: ApplicationWrapper;
}

interface TooltipPosition {
  index: number;
  rect: DOMRect;
  isFirst?: boolean;
}

class TableTooltip {
  private columnTooltip: HTMLElement;
  private rowTooltip: HTMLElement;
  private currentColumnPosition: TooltipPosition | null = null;
  private currentRowPosition: TooltipPosition | null = null;

  constructor(private view: EditorView, private options: TableTooltipOptions) {
    const renderer = options.appWrapper.getRendererFactory();
    
    // Create column tooltip
    const columnTooltipResult = renderer.components.tooltip({
      content: '',
      className: 'prosemirror-tooltip prosemirror-table-tooltip prosemirror-table-column-tooltip',
      interactive: true
    });
    this.columnTooltip = columnTooltipResult.element;
    document.body.appendChild(this.columnTooltip);
    
    // Create row tooltip
    const rowTooltipResult = renderer.components.tooltip({
      content: '',
      className: 'prosemirror-tooltip prosemirror-table-tooltip prosemirror-table-row-tooltip',
      interactive: true
    });
    this.rowTooltip = rowTooltipResult.element;
    document.body.appendChild(this.rowTooltip);

    // Start both hidden
    this.columnTooltip.classList.add('hidden');
    this.rowTooltip.classList.add('hidden');

    this.update = this.update.bind(this);
  }

  update() {
    const { state } = this.view;
    const { selection } = state;

    // Check if we're in a table
    let rect: TableRect;
    try {
      rect = selectedRect(state);
    } catch (error) {
      this.hideTooltip();
      return;
    }

    // Check if selection is within a single cell
    if (rect.left !== rect.right - 1 || rect.top !== rect.bottom - 1) {
      this.hideTooltip();
      return;
    }

    // Find the table node and its position
    let tableNode = null;
    let tablePos = -1;
    
    state.doc.descendants((node, pos) => {
      if (node.type.name === 'table' && pos < selection.from && pos + node.nodeSize > selection.from) {
        tableNode = node;
        tablePos = pos;
        return false;
      }
    });

    if (!tableNode || tablePos === -1) {
      this.hideTooltip();
      return;
    }

    // Get the table DOM element
    const tableDOM = this.view.domAtPos(tablePos + 1).node as HTMLElement;
    const table = tableDOM.closest('table');
    if (!table) {
      this.hideTooltip();
      return;
    }

    // Get cell DOM element
    const cellPos = selection.from;
    const cellDOM = this.view.domAtPos(cellPos).node as HTMLElement;
    const cell = cellDOM.nodeType === 1 ? cellDOM : cellDOM.parentElement;
    const cellElement = cell?.closest('td, th');
    
    if (!cellElement) {
      this.hideTooltip();
      return;
    }

    // Get column and row indices
    const row = cellElement.parentElement as HTMLTableRowElement;
    const columnIndex = Array.from(row.cells).indexOf(cellElement as HTMLTableCellElement);
    const rowIndex = Array.from(table.rows).indexOf(row);

    // Show column tooltip (rect.left is the column index in the table structure)
    this.showColumnTooltip(table, columnIndex, rect.left, columnIndex === 0, tablePos);

    // Also show row tooltip (rect.top is the row index in the table structure)
    this.showRowTooltip(table, rowIndex, rect.top, rowIndex === 0, tablePos);
  }

  private showColumnTooltip(table: HTMLTableElement, columnIndex: number, tableColumnIndex: number, isFirst: boolean, tablePos: number) {
    // Get the visual bounds of the column
    const cells = Array.from(table.querySelectorAll(`td:nth-child(${columnIndex + 1}), th:nth-child(${columnIndex + 1})`));
    if (cells.length === 0) return;

    const firstCell = cells[0] as HTMLElement;
    const cellRect = firstCell.getBoundingClientRect();
    
    // Check if ALL cells in this column are headers
    let isHeader = false;
    if (isFirst) {
      const { state } = this.view;
      const rect = selectedRect(state);
      if (rect) {
        const map = rect.map;
        let allHeaders = true;
        // Check ALL cells in the first column
        for (let row = 0; row < map.height; row++) {
          const pos = map.map[row * map.width]; // First column of this row (table-relative)
          // tablePos + 1 skips the table node itself, pos is relative to table content
          const absolutePos = tablePos + 1 + pos;
          const cell = state.doc.nodeAt(absolutePos);
          if (!cell || cell.type.name !== 'table_header') {
            allHeaders = false;
            break;
          }
        }
        isHeader = allHeaders;
      }
    }
    
    this.currentColumnPosition = {
      index: tableColumnIndex,
      rect: cellRect,
      isFirst: isFirst
    };

    this.updateTooltipContent('column', this.columnTooltip, isFirst, isHeader);
    this.positionTooltip(cellRect, 'column', this.columnTooltip);
    this.columnTooltip.classList.remove('hidden');
  }

  private showRowTooltip(table: HTMLTableElement, rowIndex: number, tableRowIndex: number, isFirst: boolean, tablePos: number) {
    const row = table.rows[rowIndex];
    if (!row) return;

    const rowRect = row.getBoundingClientRect();
    
    // Check if ALL cells in this row are headers
    let isHeader = false;
    if (isFirst) {
      const { state } = this.view;
      const rect = selectedRect(state);
      if (rect) {
        const map = rect.map;
        let allHeaders = true;
        // Check ALL cells in the first row
        for (let col = 0; col < map.width; col++) {
          const pos = map.map[col]; // First row of this column (table-relative)
          // tablePos + 1 skips the table node itself, pos is relative to table content
          const absolutePos = tablePos + 1 + pos;
          const cell = state.doc.nodeAt(absolutePos);
          if (!cell || cell.type.name !== 'table_header') {
            allHeaders = false;
            break;
          }
        }
        isHeader = allHeaders;
      }
    }
    
    this.currentRowPosition = {
      index: tableRowIndex,
      rect: rowRect,
      isFirst: isFirst
    };

    this.updateTooltipContent('row', this.rowTooltip, isFirst, isHeader);
    this.positionTooltip(rowRect, 'row', this.rowTooltip);
    this.rowTooltip.classList.remove('hidden');
  }

  private updateTooltipContent(type: 'column' | 'row', tooltip: HTMLElement, isFirst: boolean = false, isHeader: boolean = false) {
    const targetTooltip = tooltip;
    
    // Clear tooltip content
    targetTooltip.innerText = '';
    
    // Create content container
    const content = document.createElement('div');
    content.className = 'prosemirror-table-tooltip-content';
    
    // Create buttons safely to prevent XSS
    const insertBeforeBtn = document.createElement('button');
    insertBeforeBtn.className = 'prosemirror-table-tooltip-button insert-before';
    insertBeforeBtn.title = `Insert ${type} before`;
    insertBeforeBtn.innerHTML = '<span class="icon">+</span>';
    
    const deleteBtn = document.createElement('button');
    deleteBtn.className = 'prosemirror-table-tooltip-button delete';
    deleteBtn.title = `Delete ${type}`;
    deleteBtn.innerHTML = '<span class="icon">×</span>';
    
    content.appendChild(insertBeforeBtn);
    content.appendChild(deleteBtn);
    
    if (isFirst) {
      const toggleHeaderBtn = document.createElement('button');
      toggleHeaderBtn.className = `prosemirror-table-tooltip-button toggle-header ${isHeader ? 'active' : ''}`;
      toggleHeaderBtn.title = `Toggle ${type} header`;
      toggleHeaderBtn.innerHTML = '<span class="icon">H</span>';
      content.appendChild(toggleHeaderBtn);
    }
    
    const insertAfterBtn = document.createElement('button');
    insertAfterBtn.className = 'prosemirror-table-tooltip-button insert-after';
    insertAfterBtn.title = `Insert ${type} after`;
    insertAfterBtn.innerHTML = '<span class="icon">+</span>';
    
    content.appendChild(insertAfterBtn);
    targetTooltip.appendChild(content);

    // Add click handlers using the created button elements
    if (type === 'column') {
      insertBeforeBtn.addEventListener('click', () => this.handleColumnAction('insertBefore'));
      deleteBtn.addEventListener('click', () => this.handleColumnAction('delete'));
      insertAfterBtn.addEventListener('click', () => this.handleColumnAction('insertAfter'));
      if (isFirst) {
        const toggleHeaderBtn = content.querySelector('.toggle-header') as HTMLButtonElement;
        toggleHeaderBtn?.addEventListener('click', () => this.handleColumnAction('toggleHeader'));
      }
    } else {
      insertBeforeBtn.addEventListener('click', () => this.handleRowAction('insertBefore'));
      deleteBtn.addEventListener('click', () => this.handleRowAction('delete'));
      insertAfterBtn.addEventListener('click', () => this.handleRowAction('insertAfter'));
      if (isFirst) {
        const toggleHeaderBtn = content.querySelector('.toggle-header') as HTMLButtonElement;
        toggleHeaderBtn?.addEventListener('click', () => this.handleRowAction('toggleHeader'));
      }
    }
  }

  private positionTooltip(rect: DOMRect, type: 'column' | 'row', tooltip: HTMLElement) {
    const targetTooltip = tooltip;
    const scrollX = window.pageXOffset || document.documentElement.scrollLeft;
    const scrollY = window.pageYOffset || document.documentElement.scrollTop;
    
    if (type === 'column') {
      // Position above the column, centered
      const left = rect.left + scrollX;
      const top = rect.top + scrollY;
      
      targetTooltip.style.setProperty('--tooltip-left', `${left}px`);
      targetTooltip.style.setProperty('--tooltip-top', `${top}px`);
      targetTooltip.style.setProperty('--tooltip-transform', 'translate(0, -100%)');
    } else {
      // Position to the left of the row, centered
      const left = rect.left + scrollX;
      const top = rect.top + scrollY;
      
      targetTooltip.style.setProperty('--tooltip-left', `${left}px`);
      targetTooltip.style.setProperty('--tooltip-top', `${top}px`);
      targetTooltip.style.setProperty('--tooltip-transform', 'translate(-100%, 0)');
    }
  }

  private handleColumnAction(action: 'insertBefore' | 'delete' | 'insertAfter' | 'toggleHeader') {
    if (!this.currentColumnPosition) return;

    // The prosemirror-tables commands work on the current selection
    // They automatically handle operating on all cells in the column/row
    const { state, dispatch } = this.view;
    
    switch (action) {
      case 'insertBefore':
        if (addColumnBefore(state, dispatch)) {
          this.view.focus();
        }
        break;
      case 'delete':
        if (deleteColumn(state, dispatch)) {
          this.view.focus();
        }
        break;
      case 'insertAfter':
        if (addColumnAfter(state, dispatch)) {
          this.view.focus();
        }
        break;
      case 'toggleHeader':
        this.toggleColumnHeader();
        break;
    }
  }

  private handleRowAction(action: 'insertBefore' | 'delete' | 'insertAfter' | 'toggleHeader') {
    if (!this.currentRowPosition) return;

    // The prosemirror-tables commands work on the current selection
    // They automatically handle operating on all cells in the row
    const { state, dispatch } = this.view;
    
    switch (action) {
      case 'insertBefore':
        if (addRowBefore(state, dispatch)) {
          this.view.focus();
        }
        break;
      case 'delete':
        if (deleteRow(state, dispatch)) {
          this.view.focus();
        }
        break;
      case 'insertAfter':
        if (addRowAfter(state, dispatch)) {
          this.view.focus();
        }
        break;
      case 'toggleHeader':
        this.toggleRowHeader();
        break;
    }
  }

  private toggleColumnHeader() {
    if (!this.currentColumnPosition || !this.currentColumnPosition.isFirst) return;

    const { state, dispatch } = this.view;
    const rect = selectedRect(state);
    if (!rect) return;

    const { tr } = state;
    const map = rect.map;
    
    // Find the table position
    let tablePos = -1;
    state.doc.descendants((node, pos) => {
      if (node.type.name === 'table' && pos < state.selection.from && pos + node.nodeSize > state.selection.from) {
        tablePos = pos;
        return false;
      }
    });
    
    if (tablePos === -1) return;
    
    // Check if ALL cells in the first column are headers
    let allHeaders = true;
    for (let row = 0; row < map.height; row++) {
      const pos = map.map[row * map.width];
      const absolutePos = tablePos + 1 + pos;
      const cell = state.doc.nodeAt(absolutePos);
      if (!cell || cell.type.name !== 'table_header') {
        allHeaders = false;
        break;
      }
    }
    
    // Toggle cells in the first column
    for (let row = 0; row < map.height; row++) {
      const pos = map.map[row * map.width];
      const absolutePos = tablePos + 1 + pos;
      const cell = tr.doc.nodeAt(absolutePos);
      if (!cell) continue;
      
      if (allHeaders) {
        // Removing headers - check if this cell is also part of a header row
        if (row === 0) {
          // Check if the entire first row is headers
          let firstRowAllHeaders = true;
          for (let col = 0; col < map.width; col++) {
            const rowPos = map.map[col];
            const rowAbsolutePos = tablePos + 1 + rowPos;
            const rowCell = state.doc.nodeAt(rowAbsolutePos);
            if (!rowCell || rowCell.type.name !== 'table_header') {
              firstRowAllHeaders = false;
              break;
            }
          }
          // If first row is all headers, keep this cell as header
          if (firstRowAllHeaders) continue;
        }
        // Convert to regular cell
        tr.setNodeMarkup(absolutePos, state.schema.nodes.table_cell, cell.attrs);
      } else {
        // Convert to header
        tr.setNodeMarkup(absolutePos, state.schema.nodes.table_header, cell.attrs);
      }
    }
    
    dispatch(tr);
    this.view.focus();
  }

  private toggleRowHeader() {
    if (!this.currentRowPosition || !this.currentRowPosition.isFirst) return;

    const { state, dispatch } = this.view;
    const rect = selectedRect(state);
    if (!rect) return;

    const { tr } = state;
    const map = rect.map;
    
    // Find the table position
    let tablePos = -1;
    state.doc.descendants((node, pos) => {
      if (node.type.name === 'table' && pos < state.selection.from && pos + node.nodeSize > state.selection.from) {
        tablePos = pos;
        return false;
      }
    });
    
    if (tablePos === -1) return;
    
    // Check if ALL cells in the first row are headers
    let allHeaders = true;
    for (let col = 0; col < map.width; col++) {
      const pos = map.map[col];
      const absolutePos = tablePos + 1 + pos;
      const cell = state.doc.nodeAt(absolutePos);
      if (!cell || cell.type.name !== 'table_header') {
        allHeaders = false;
        break;
      }
    }
    
    // Toggle cells in the first row
    for (let col = 0; col < map.width; col++) {
      const pos = map.map[col];
      const absolutePos = tablePos + 1 + pos;
      const cell = tr.doc.nodeAt(absolutePos);
      if (!cell) continue;
      
      if (allHeaders) {
        // Removing headers - check if this cell is also part of a header column
        if (col === 0) {
          // Check if the entire first column is headers
          let firstColAllHeaders = true;
          for (let row = 0; row < map.height; row++) {
            const colPos = map.map[row * map.width];
            const colAbsolutePos = tablePos + 1 + colPos;
            const colCell = state.doc.nodeAt(colAbsolutePos);
            if (!colCell || colCell.type.name !== 'table_header') {
              firstColAllHeaders = false;
              break;
            }
          }
          // If first column is all headers, keep this cell as header
          if (firstColAllHeaders) continue;
        }
        // Convert to regular cell
        tr.setNodeMarkup(absolutePos, state.schema.nodes.table_cell, cell.attrs);
      } else {
        // Convert to header
        tr.setNodeMarkup(absolutePos, state.schema.nodes.table_header, cell.attrs);
      }
    }
    
    dispatch(tr);
    this.view.focus();
  }

  private hideTooltip() {
    this.columnTooltip.classList.add('hidden');
    this.rowTooltip.classList.add('hidden');
    this.currentColumnPosition = null;
    this.currentRowPosition = null;
  }

  destroy() {
    this.columnTooltip.remove();
    this.rowTooltip.remove();
  }
}

export function createTablePlugin(appWrapper: ApplicationWrapper): Plugin {
  return new Plugin({
    view(editorView) {
      const tooltip = new TableTooltip(editorView, { appWrapper });
      return {
        update: () => tooltip.update(),
        destroy: () => tooltip.destroy()
      };
    }
  });
}
