import { 
  ElementAttributes, 
  ElementRenderer, 
  RenderResult,
  InputAttributes,
  SelectOption,
  ButtonAttributes
} from './interfaces';

export abstract class BaseElementRenderer implements ElementRenderer {
  protected applyAttributes(element: HTMLElement, attrs?: ElementAttributes): void {
    if (!attrs) return;
    
    if (attrs.id) element.id = attrs.id;
    if (attrs.className) element.className = attrs.className;
    
    if (attrs.style) {
      Object.entries(attrs.style).forEach(([key, value]) => {
        if (value !== undefined) {
          (element.style as any)[key] = value;
        }
      });
    }
    
    if (attrs.dataset) {
      Object.entries(attrs.dataset).forEach(([key, value]) => {
        element.dataset[key] = value;
      });
    }
    
    if (attrs.attributes) {
      Object.entries(attrs.attributes).forEach(([key, value]) => {
        element.setAttribute(key, value);
      });
    }
    
    if (attrs.listeners) {
      Object.entries(attrs.listeners).forEach(([event, handler]) => {
        element.addEventListener(event, handler);
      });
    }
  }
  
  protected createRenderResult<T extends HTMLElement>(
    element: T, 
    listeners?: Record<string, EventListener>
  ): RenderResult<T> {
    const eventListeners = listeners ? Object.entries(listeners) : [];
    
    return {
      element,
      destroy: () => {
        // Remove event listeners
        eventListeners.forEach(([event, handler]) => {
          element.removeEventListener(event, handler);
        });
        // Remove from DOM
        element.remove();
      },
      update: (newAttrs: any) => {
        this.applyAttributes(element, newAttrs);
      }
    };
  }
  
  abstract div(attrs?: ElementAttributes): RenderResult<HTMLDivElement>;
  abstract span(attrs?: ElementAttributes): RenderResult<HTMLSpanElement>;
  abstract input(attrs?: InputAttributes): RenderResult<HTMLInputElement>;
  abstract button(text: string, attrs?: ButtonAttributes): RenderResult<HTMLButtonElement>;
  abstract label(text: string, attrs?: ElementAttributes & { for?: string }): RenderResult<HTMLLabelElement>;
  abstract select(options: SelectOption[], attrs?: ElementAttributes): RenderResult<HTMLSelectElement>;
  abstract form(attrs?: ElementAttributes): RenderResult<HTMLFormElement>;
} 