import { AutocompleteConfig, AutocompleteItem, RenderResult } from '../interfaces';

/**
 * DOMAutocompleteRenderer is a renderer for the autocomplete component.
 * 
 * @todo: Move styles to a separate file.
 */
export class DOMAutocompleteRenderer {
  create(config: AutocompleteConfig): RenderResult<HTMLElement> {
    // Create container
    const container = document.createElement('div');
    container.className = 'autocomplete-container';
    container.style.position = 'relative';
    
    // Create input
    const input = document.createElement('input');
    input.type = 'text';
    input.placeholder = config.placeholder || '';
    input.value = config.value || '';
    input.required = !!config.required;
    input.className = 'autocomplete-input form-text';
    
    // Create dropdown
    const dropdown = document.createElement('ul');
    dropdown.className = 'autocomplete-dropdown';
    dropdown.style.position = 'fixed'; // Changed to fixed for proper modal overlay
    dropdown.style.background = 'white';
    dropdown.style.border = '1px solid #ccc';
    dropdown.style.borderRadius = '4px';
    dropdown.style.maxHeight = '200px';
    dropdown.style.overflowY = 'auto';
    dropdown.style.display = 'none';
    dropdown.style.zIndex = '100000'; // High z-index to appear above modals
    dropdown.style.listStyle = 'none';
    dropdown.style.margin = '0';
    dropdown.style.padding = '0';
    dropdown.style.boxShadow = '0 2px 4px rgba(0,0,0,0.2)';
    
    // Create loading indicator
    const loading = document.createElement('div');
    loading.className = 'autocomplete-loading';
    loading.textContent = 'Loading...';
    loading.style.visibility = 'hidden';
    loading.style.padding = '8px';
    loading.style.color = '#666';
    
    container.appendChild(input);
    container.appendChild(loading);
    // Append dropdown to body for proper positioning
    document.body.appendChild(dropdown);
    
    let currentItems: AutocompleteItem[] = [];
    let selectedIndex = -1;
    let debounceTimer: any;
    
    // Helper to position dropdown
    const positionDropdown = () => {
      const rect = input.getBoundingClientRect();
      const dropdownHeight = dropdown.offsetHeight;
      const spaceBelow = window.innerHeight - rect.bottom;
      const spaceAbove = rect.top;
      
      // Determine if we should show above or below
      if (spaceBelow < dropdownHeight && spaceAbove > spaceBelow) {
        // Show above
        dropdown.style.bottom = `${window.innerHeight - rect.top}px`;
        dropdown.style.top = 'auto';
      } else {
        // Show below
        dropdown.style.top = `${rect.bottom}px`;
        dropdown.style.bottom = 'auto';
      }
      
      dropdown.style.left = `${rect.left}px`;
      dropdown.style.width = `${rect.width}px`;
    };
    
    // Helper to render dropdown items
    const renderDropdown = (items: AutocompleteItem[]) => {
      dropdown.innerText = '';
      selectedIndex = -1;
      
      if (items.length === 0) {
        dropdown.style.display = 'none';
        return;
      }
      
      // Group items if needed
      const grouped: Map<string, AutocompleteItem[]> = new Map();
      items.forEach(item => {
        const group = item.group || '';
        if (!grouped.has(group)) {
          grouped.set(group, []);
        }
        grouped.get(group)!.push(item);
      });
      
      // Render grouped items
      let itemIndex = 0;
      grouped.forEach((groupItems, groupName) => {
        if (groupName) {
          const groupHeader = document.createElement('li');
          groupHeader.className = 'autocomplete-group-header';
          groupHeader.textContent = groupName;
          groupHeader.style.padding = '4px 8px';
          groupHeader.style.fontWeight = 'bold';
          groupHeader.style.color = '#666';
          dropdown.appendChild(groupHeader);
        }
        
        groupItems.forEach(item => {
          const li = document.createElement('li');
          li.className = 'autocomplete-item';
          li.style.padding = '8px';
          li.style.cursor = 'pointer';
          li.dataset.index = String(itemIndex++);
          
          const label = document.createElement('div');
          label.className = 'autocomplete-item-label';
          label.textContent = item.label;
          li.appendChild(label);
          
          if (item.description) {
            const desc = document.createElement('div');
            desc.className = 'autocomplete-item-description';
            desc.textContent = item.description;
            desc.style.fontSize = '0.9em';
            desc.style.color = '#666';
            li.appendChild(desc);
          }
          
          li.addEventListener('mouseenter', () => {
            highlightItem(parseInt(li.dataset.index || '0'));
          });
          
          li.addEventListener('click', (e) => {
            e.preventDefault();
            selectItem(item);
          });
          
          dropdown.appendChild(li);
        });
      });
      
      dropdown.style.display = 'block';
      positionDropdown();
    };
    
    // Helper to highlight item
    const highlightItem = (index: number) => {
      const items = dropdown.querySelectorAll('.autocomplete-item');
      items.forEach((item, i) => {
        if (i === index) {
          item.classList.add('highlighted');
          (item as HTMLElement).style.background = '#f0f0f0';
        } else {
          item.classList.remove('highlighted');
          (item as HTMLElement).style.background = '';
        }
      });
      selectedIndex = index;
    };
    
    // Helper to select item
    const selectItem = (item: AutocompleteItem) => {
      input.value = item.value;
      dropdown.style.display = 'none';
      config.onSelect(item);
    };
    
    // Input event handler
    input.addEventListener('input', async () => {
      const term = input.value;
      
      if (!term) {
        dropdown.style.display = 'none';
        return;
      }
      
      // Debounce the search
      clearTimeout(debounceTimer);
      debounceTimer = setTimeout(async () => {
        loading.style.visibility = 'visible';
        dropdown.style.display = 'none';
        
        try {
          const items = await config.onQuery(term);
          currentItems = items;
          renderDropdown(items);
        } catch (error) {
          console.error('Autocomplete query error:', error);
          dropdown.style.display = 'none';
        } finally {
          loading.style.visibility = 'hidden';
        }
      }, 300);
    });
    
    // Keyboard navigation
    input.addEventListener('keydown', (e) => {
      if (dropdown.style.display === 'none') return;
      
      switch (e.key) {
        case 'ArrowDown':
          e.preventDefault();
          highlightItem(Math.min(selectedIndex + 1, currentItems.length - 1));
          break;
          
        case 'ArrowUp':
          e.preventDefault();
          highlightItem(Math.max(selectedIndex - 1, -1));
          break;
          
        case 'Enter':
          e.preventDefault();
          if (selectedIndex >= 0 && selectedIndex < currentItems.length) {
            selectItem(currentItems[selectedIndex]);
          }
          break;
          
        case 'Escape':
          dropdown.style.display = 'none';
          break;
      }
    });
    
    // Close dropdown on blur
    input.addEventListener('blur', () => {
      // Delay to allow click events to fire
      setTimeout(() => {
        dropdown.style.display = 'none';
      }, 200);
    });
    
    // Focus handler
    input.addEventListener('focus', () => {
      if (input.value && currentItems.length > 0) {
        dropdown.style.display = 'block';
        positionDropdown();
      }
    });
    
    // Reposition on scroll/resize
    window.addEventListener('scroll', () => {
      if (dropdown.style.display !== 'none') {
        positionDropdown();
      }
    }, true);
    
    window.addEventListener('resize', () => {
      if (dropdown.style.display !== 'none') {
        positionDropdown();
      }
    });
    
    return {
      element: container,
      destroy: () => {
        clearTimeout(debounceTimer);
        dropdown.remove(); // Remove dropdown from body
        container.remove();
      },
      update: (newConfig: Partial<AutocompleteConfig>) => {
        if (newConfig.value !== undefined) {
          input.value = newConfig.value;
        }
        if (newConfig.placeholder !== undefined) {
          input.placeholder = newConfig.placeholder;
        }
        if (newConfig.required !== undefined) {
          input.required = newConfig.required;
        }
      }
    };
  }
}
