import { 
  ComponentRenderer, 
  TooltipConfig, 
  DialogConfig, 
  AutocompleteConfig,
  RenderResult 
} from '../interfaces';
import { DOMTooltipRenderer } from './dom-tooltip-renderer';
import { DOMAutocompleteRenderer } from './dom-autocomplete-renderer';

/**
 * DOMComponentRenderer is a renderer for the component component.
 *
 * @todo: Move styles to a separate file.
 */
export class DOMComponentRenderer implements ComponentRenderer {
  private tooltipRenderer = new DOMTooltipRenderer();
  private autocompleteRenderer = new DOMAutocompleteRenderer();
  
  tooltip(config: TooltipConfig): RenderResult<HTMLElement> {
    return this.tooltipRenderer.create(config);
  }
  
  dialog(config: DialogConfig): RenderResult<HTMLElement> & { open: () => void; close: () => void } {
    // Create dialog elements
    const overlay = document.createElement('div');
    overlay.className = 'prosemirror-dialog-overlay';
    overlay.style.position = 'fixed';
    overlay.style.top = '0';
    overlay.style.left = '0';
    overlay.style.width = '100%';
    overlay.style.height = '100%';
    overlay.style.backgroundColor = 'rgba(0, 0, 0, 0.5)';
    overlay.style.zIndex = '999';
    overlay.style.display = 'none';
    
    const dialog = document.createElement('div');
    dialog.className = 'prosemirror-dialog';
    dialog.style.position = 'fixed';
    dialog.style.top = '50%';
    dialog.style.left = '50%';
    dialog.style.transform = 'translate(-50%, -50%)';
    dialog.style.backgroundColor = 'white';
    dialog.style.padding = '20px';
    dialog.style.borderRadius = '8px';
    dialog.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.1)';
    dialog.style.zIndex = '1000';
    dialog.style.display = 'none';
    dialog.style.maxWidth = '90vw';
    dialog.style.maxHeight = '90vh';
    dialog.style.overflow = 'auto';
    
    if (config.width) {
      dialog.style.width = typeof config.width === 'number' 
        ? `${config.width}px` 
        : config.width;
    }
    
    if (config.height) {
      dialog.style.height = typeof config.height === 'number' 
        ? `${config.height}px` 
        : config.height;
    }
    
    // Create header
    const header = document.createElement('div');
    header.className = 'prosemirror-dialog-header';
    header.style.display = 'flex';
    header.style.justifyContent = 'space-between';
    header.style.alignItems = 'center';
    header.style.marginBottom = '16px';
    
    const title = document.createElement('h2');
    title.className = 'prosemirror-dialog-title';
    title.textContent = config.title;
    title.style.margin = '0';
    
    const closeButton = document.createElement('button');
    closeButton.className = 'prosemirror-dialog-close';
    closeButton.innerHTML = '&times;';
    closeButton.style.border = 'none';
    closeButton.style.background = 'none';
    closeButton.style.fontSize = '24px';
    closeButton.style.cursor = 'pointer';
    closeButton.style.padding = '0';
    closeButton.style.width = '30px';
    closeButton.style.height = '30px';
    
    header.appendChild(title);
    header.appendChild(closeButton);
    
    // Create content area
    const content = document.createElement('div');
    content.className = 'prosemirror-dialog-content';
    content.appendChild(config.content);
    
    // Create footer if buttons are provided
    let footer: HTMLElement | null = null;
    if (config.buttons && config.buttons.length > 0) {
      footer = document.createElement('div');
      footer.className = 'prosemirror-dialog-footer';
      footer.style.marginTop = '16px';
      footer.style.display = 'flex';
      footer.style.justifyContent = 'flex-end';
      footer.style.gap = '8px';
      
      config.buttons.forEach(btn => {
        const button = document.createElement('button');
        button.className = btn.primary ? 'button button--primary' : 'button button--secondary';
        button.textContent = btn.text;
        button.addEventListener('click', btn.action);
        footer!.appendChild(button);
      });
    }
    
    // Assemble dialog
    dialog.appendChild(header);
    dialog.appendChild(content);
    if (footer) {
      dialog.appendChild(footer);
    }
    
    // Add to document
    document.body.appendChild(overlay);
    document.body.appendChild(dialog);
    
    // Dialog control functions
    const open = () => {
      overlay.style.display = 'block';
      dialog.style.display = 'block';
      // Focus first input if available
      const firstInput = dialog.querySelector('input, textarea, select');
      if (firstInput) {
        (firstInput as HTMLElement).focus();
      }
    };
    
    const close = () => {
      overlay.style.display = 'none';
      dialog.style.display = 'none';
    };
    
    // Event handlers
    closeButton.addEventListener('click', close);
    
    if (config.modal !== false) {
      overlay.addEventListener('click', close);
      
      // Prevent closing when clicking inside dialog
      dialog.addEventListener('click', (e) => {
        e.stopPropagation();
      });
    }
    
    // ESC key handler
    const escHandler = (e: KeyboardEvent) => {
      if (e.key === 'Escape' && dialog.style.display !== 'none') {
        close();
      }
    };
    document.addEventListener('keydown', escHandler);
    
    return {
      element: dialog,
      open,
      close,
      destroy: () => {
        document.removeEventListener('keydown', escHandler);
        overlay.remove();
        dialog.remove();
      },
      update: (newConfig: Partial<DialogConfig>) => {
        if (newConfig.title !== undefined) {
          title.textContent = newConfig.title;
        }
        if (newConfig.content !== undefined) {
          content.innerText = '';
          content.appendChild(newConfig.content);
        }
      }
    };
  }
  
  autocomplete(config: AutocompleteConfig): RenderResult<HTMLElement> {
    return this.autocompleteRenderer.create(config);
  }
} 