import { Mark, MarkType } from "prosemirror-model";
import { EditorView } from "prosemirror-view";

/**
 * Find a specific mark type around a given position and return its complete range
 * 
 * This utility searches for a mark in a range around the given position,
 * which is useful when dealing with overlapping marks or when the exact
 * position might not have the mark but nearby positions do.
 * 
 * @param view - The ProseMirror editor view
 * @param pos - The position to search around
 * @param markType - The type of mark to find
 * @param searchRange - How many positions to search in each direction (default: 50)
 * @returns Object with mark, from, and to properties, or null if not found
 */
export function findMarkAtPosition(
  view: EditorView, 
  pos: number, 
  markType: MarkType, 
  searchRange: number = 50
): { mark: Mark, from: number, to: number } | null {
  if (pos < 0) return null;

  try {
    // Search for the mark in a range around the given position
    const startSearch = Math.max(0, pos - searchRange);
    const endSearch = Math.min(view.state.doc.content.size, pos + searchRange);
    
    for (let searchPos = startSearch; searchPos <= endSearch; searchPos++) {
      try {
        const resolved = view.state.doc.resolve(searchPos);
        const marks = resolved.marks();
        const foundMark = marks.find(m => m.type === markType);
        
        if (foundMark) {
          // Find the complete range of this mark
          let markFrom: number | null = null;
          let markTo: number | null = null;
          
          // Walk through the document to find all text nodes with this exact mark
          view.state.doc.descendants((node, nodePos) => {
            if (node.isText && node.marks.some(mark => mark.eq(foundMark))) {
              const nodeStart = nodePos;
              const nodeEnd = nodePos + node.nodeSize;
              
              if (markFrom === null || nodeStart < markFrom) {
                markFrom = nodeStart;
              }
              if (markTo === null || nodeEnd > markTo) {
                markTo = nodeEnd;
              }
            }
            return true;
          });
          
          // If descendants approach didn't work, use fallback method
          if (markFrom === null || markTo === null) {
            markFrom = searchPos;
            markTo = searchPos;
            
            // Search backward to find the start of the mark
            while (markFrom > 0) {
              try {
                const prevResolved = view.state.doc.resolve(markFrom - 1);
                const prevMarks = prevResolved.marks();
                const prevMark = prevMarks.find(m => m.type === markType);
                
                if (!prevMark || !prevMark.eq(foundMark)) {
                  break;
                }
                markFrom--;
              } catch {
                break;
              }
            }
            
            // Search forward to find the end of the mark
            const docSize = view.state.doc.content.size;
            while (markTo < docSize) {
              try {
                const nextResolved = view.state.doc.resolve(markTo + 1);
                const nextMarks = nextResolved.marks();
                const nextMark = nextMarks.find(m => m.type === markType);
                
                if (!nextMark || !nextMark.eq(foundMark)) {
                  break;
                }
                markTo++;
              } catch {
                break;
              }
            }
          }
          
          if (markFrom !== null && markTo !== null) {
            return { mark: foundMark, from: markFrom, to: markTo };
          }
        }
      } catch {
        // Continue searching if this position is invalid
        continue;
      }
    }
  } catch (error) {
    console.error('Error finding mark at position:', error);
  }
  
  return null;
}
