<?php

namespace Drupal\prosemirror\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\prosemirror\ProseMirrorElementInterface;

/**
 * Defines the ProseMirror element entity.
 *
 * @ConfigEntityType(
 *   id = "prosemirror_element",
 *   label = @Translation("ProseMirror element"),
 *   handlers = {
 *     "list_builder" = "Drupal\prosemirror\ProseMirrorElementListBuilder",
 *     "form" = {
 *       "add" = "Drupal\prosemirror\Form\ProseMirrorElementForm",
 *       "edit" = "Drupal\prosemirror\Form\ProseMirrorElementForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider"
 *     }
 *   },
 *   admin_permission = "administer prosemirror",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "type",
 *     "css_classes",
 *     "content",
 *     "content_min",
 *     "content_max",
 *     "groups",
 *     "options"
 *   },
 *   links = {
 *     "add-form" = "/admin/config/content/prosemirror/elements/add",
 *     "edit-form" = "/admin/config/content/prosemirror/elements/{prosemirror_element}/edit",
 *     "delete-form" = "/admin/config/content/prosemirror/elements/{prosemirror_element}/delete",
 *     "collection" = "/admin/config/content/prosemirror/elements"
 *   }
 * )
 */
class ProseMirrorElement extends ConfigEntityBase implements ProseMirrorElementInterface {

  /**
   * The element ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The element label.
   *
   * @var string
   */
  protected $label;

  /**
   * The element type plugin ID.
   *
   * @var string
   */
  protected $type;

  /**
   * The CSS classes for this element.
   *
   * @var array
   */
  protected $css_classes = [];

  /**
   * The content reference (element type or element group).
   *
   * @var string
   */
  protected $content;

  /**
   * The minimum content count.
   *
   * @var int|null
   */
  protected $content_min;

  /**
   * The maximum content count.
   *
   * @var int|null
   */
  protected $content_max;

  /**
   * The element groups this element belongs to.
   *
   * @var array
   */
  protected $groups = [];

  /**
   * Additional options specific to the element type.
   *
   * @var array
   */
  protected $options = [];

  /**
   * {@inheritdoc}
   */
  public function getType() {
    return $this->type;
  }

  /**
   * {@inheritdoc}
   */
  public function setType($type) {
    $this->type = $type;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCssClasses() {
    return $this->css_classes;
  }

  /**
   * {@inheritdoc}
   */
  public function setCssClasses(array $css_classes) {
    $this->css_classes = $css_classes;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getContent() {
    return $this->content;
  }

  /**
   * {@inheritdoc}
   */
  public function setContent($content) {
    $this->content = $content;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getContentMin() {
    return $this->content_min;
  }

  /**
   * {@inheritdoc}
   */
  public function getContentMax() {
    return $this->content_max;
  }

  /**
   * {@inheritdoc}
   */
  public function getGroups() {
    return $this->groups;
  }

  /**
   * {@inheritdoc}
   */
  public function setGroups(array $groups) {
    $this->groups = $groups;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getContentExpression() {
    $min = $this->content_min;
    $max = $this->content_max;

    if ($min === NULL && $max === NULL) {
      return '*';
    }
    elseif ($min === 1 && $max === NULL) {
      return '+';
    }
    elseif ($min !== NULL && $max !== NULL) {
      return '{' . $min . ',' . $max . '}';
    }
    elseif ($min !== NULL) {
      return '{' . $min . ',}';
    }
    else {
      return '{0,' . $max . '}';
    }
  }

  /**
   * {@inheritdoc}
   */
  public function setContentExpression($content_expression) {
    $this->content_expression = $content_expression;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOptions() {
    return ($this->options ?? []) + [
      'element_id' => $this->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setOptions(array $options) {
    $this->options = $options;
    return $this;
  }

}
