<?php

namespace Drupal\prosemirror\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form handler for the ProseMirror element group add/edit forms.
 */
class ProseMirrorElementGroupForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\prosemirror\ProseMirrorElementGroupInterface $element_group */
    $element_group = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $element_group->label(),
      '#description' => $this->t('Name of the ProseMirror element group.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $element_group->id(),
      '#machine_name' => [
        'exists' => '\Drupal\prosemirror\Entity\ProseMirrorElementGroup::load',
      ],
      '#disabled' => !$element_group->isNew(),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Additional validation for group ID.
    $id = $form_state->getValue('id');
    if ($this->entity->isNew() && $id) {
      // Check for reserved names.
      $reserved_names = ['paragraph', 'inline', 'text'];
      if (in_array($id, $reserved_names, TRUE)) {
        $form_state->setErrorByName('id', $this->t('The machine name "@id" is reserved and cannot be used.', ['@id' => $id]));
      }

      // Check if any elements are using this group ID as a content reference.
      $element_storage = $this->entityTypeManager->getStorage('prosemirror_element');
      $elements = $element_storage->loadByProperties(['content' => $id]);
      if (!empty($elements)) {
        $form_state->setErrorByName('id', $this->t('The machine name "@id" conflicts with an existing element content reference.', ['@id' => $id]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $element_group = $this->entity;
    $status = $element_group->save();

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('Created the %label ProseMirror element group.', [
          '%label' => $element_group->label(),
        ]));
        break;

      default:
        $this->messenger()->addStatus($this->t('Saved the %label ProseMirror element group.', [
          '%label' => $element_group->label(),
        ]));
    }
    $form_state->setRedirectUrl($element_group->toUrl('collection'));
  }

}
