<?php

namespace Drupal\prosemirror\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form for adding/editing ProseMirror marks.
 */
class ProseMirrorMarkForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\prosemirror\ProseMirrorMarkInterface $mark */
    $mark = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $mark->label(),
      '#description' => $this->t('Label for the mark.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $mark->id(),
      '#machine_name' => [
        'exists' => '\Drupal\prosemirror\Entity\ProseMirrorMark::load',
      ],
      '#disabled' => !$mark->isNew(),
    ];

    // Parse DOM rules.
    $form['parse_dom'] = [
      '#type' => 'details',
      '#title' => $this->t('Parse DOM Rules'),
      '#description' => $this->t('Define how this mark should be parsed from HTML/DOM.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $parse_dom_rules = $mark->getParseDom();
    $num_rules = $form_state->get('num_parse_dom_rules');
    if ($num_rules === NULL) {
      $num_rules = !empty($parse_dom_rules) ? count($parse_dom_rules) : 1;
      $form_state->set('num_parse_dom_rules', $num_rules);
    }

    $form['parse_dom']['#prefix'] = '<div id="parse-dom-wrapper">';
    $form['parse_dom']['#suffix'] = '</div>';

    for ($i = 0; $i < $num_rules; $i++) {
      $rule = $parse_dom_rules[$i] ?? [];

      $form['parse_dom'][$i] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Rule @num', ['@num' => $i + 1]),
      ];

      $form['parse_dom'][$i]['type'] = [
        '#type' => 'select',
        '#title' => $this->t('Rule type'),
        '#options' => [
          'tag' => $this->t('Tag'),
          'style' => $this->t('Style'),
          'clearMark' => $this->t('Clear mark'),
        ],
        '#default_value' => isset($rule['tag']) ? 'tag' : (isset($rule['style']) ? 'style' : 'clearMark'),
        '#required' => TRUE,
      ];

      $form['parse_dom'][$i]['tag'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Tag name'),
        '#default_value' => $rule['tag'] ?? '',
        '#states' => [
          'visible' => [
            ':input[name="parse_dom[' . $i . '][type]"]' => ['value' => 'tag'],
          ],
          'required' => [
            ':input[name="parse_dom[' . $i . '][type]"]' => ['value' => 'tag'],
          ],
        ],
      ];

      $form['parse_dom'][$i]['style'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Style rule'),
        '#description' => $this->t('E.g., "font-style=italic" or "font-weight=bold"'),
        '#default_value' => $rule['style'] ?? '',
        '#states' => [
          'visible' => [
            ':input[name="parse_dom[' . $i . '][type]"]' => ['!value' => 'tag'],
          ],
          'required' => [
            ':input[name="parse_dom[' . $i . '][type]"]' => ['value' => 'style'],
          ],
        ],
      ];

      $form['parse_dom'][$i]['get_attrs'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Attribute validation'),
        '#description' => $this->t('For tags with specific attributes. E.g., for strong tag checking font-weight.'),
        '#default_value' => $rule['get_attrs'] ?? '',
        '#states' => [
          'visible' => [
            ':input[name="parse_dom[' . $i . '][type]"]' => ['value' => 'tag'],
          ],
        ],
      ];

      $form['parse_dom'][$i]['clear_mark'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Clear mark condition'),
        '#description' => $this->t('Mark type name to clear. E.g., "em" or "strong"'),
        '#default_value' => $rule['clear_mark'] ?? '',
        '#states' => [
          'visible' => [
            ':input[name="parse_dom[' . $i . '][type]"]' => ['value' => 'clearMark'],
          ],
          'required' => [
            ':input[name="parse_dom[' . $i . '][type]"]' => ['value' => 'clearMark'],
          ],
        ],
      ];

      if ($i > 0) {
        $form['parse_dom'][$i]['remove'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove rule'),
          '#name' => 'remove_rule_' . $i,
          '#submit' => ['::removeRule'],
          '#ajax' => [
            'callback' => '::parseRulesCallback',
            'wrapper' => 'parse-dom-wrapper',
          ],
          '#limit_validation_errors' => [],
        ];
      }
    }

    $form['parse_dom']['add_rule'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another rule'),
      '#submit' => ['::addRule'],
      '#ajax' => [
        'callback' => '::parseRulesCallback',
        'wrapper' => 'parse-dom-wrapper',
      ],
      '#limit_validation_errors' => [],
    ];

    // To DOM specification.
    $form['to_dom'] = [
      '#type' => 'details',
      '#title' => $this->t('To DOM Specification'),
      '#description' => $this->t('Define how this mark should be rendered to DOM.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $to_dom = $mark->getToDom();

    $form['to_dom']['tag'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tag name'),
      '#default_value' => $to_dom['tag'] ?? '',
      '#required' => TRUE,
    ];

    $form['to_dom']['attrs'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Attributes'),
      '#description' => $this->t('One attribute per line in format: name=value'),
      '#default_value' => $this->attributesToString($to_dom['attrs'] ?? []),
    ];

    // Additional attributes.
    $form['attributes'] = [
      '#type' => 'details',
      '#title' => $this->t('Additional Attributes'),
      '#description' => $this->t('Additional mark attributes like "code: true" for code marks.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $attributes = $mark->getAttributes();

    $form['attributes']['code'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Code mark'),
      '#description' => $this->t('Check if this is a code mark (enables special behavior).'),
      '#default_value' => $attributes['code'] ?? FALSE,
    ];

    // Variants (only for link marks right now)
    if ($mark->id() === 'link' || $mark->isNew()) {
      $form['variants'] = [
        '#type' => 'details',
        '#title' => $this->t('Link Variants'),
        '#description' => $this->t('Configure link style variants (only applicable to link marks).'),
        '#open' => $mark->id() === 'link',
        '#tree' => TRUE,
        '#states' => [
          'visible' => [
            ':input[name="id"]' => ['value' => 'link'],
          ],
        ],
      ];

      $variants = $mark->getVariants();
      $variants_string = '';
      if (is_array($variants)) {
        $lines = [];
        foreach ($variants as $key => $label) {
          $lines[] = $key . '|' . $label;
        }
        $variants_string = implode("\n", $lines);
      }

      $form['variants']['variants_text'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Available Variants'),
        '#default_value' => $variants_string,
        '#description' => $this->t('Enter one variant per line in the format: <code>key|label</code><br><strong>Examples:</strong><br><code>default|Default</code><br><code>button|Button</code><br><code>external|External</code>'),
        '#rows' => 6,
      ];
    }

    return $form;
  }

  /**
   * Callback for add/remove rule buttons.
   */
  public function parseRulesCallback(array &$form, FormStateInterface $form_state) {
    return $form['parse_dom'];
  }

  /**
   * Submit handler for adding a rule.
   */
  public function addRule(array &$form, FormStateInterface $form_state) {
    $num_rules = $form_state->get('num_parse_dom_rules');
    $form_state->set('num_parse_dom_rules', $num_rules + 1);
    $form_state->setRebuild();
  }

  /**
   * Submit handler for removing a rule.
   */
  public function removeRule(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    if (preg_match('/remove_rule_(\d+)/', $triggering_element['#name'], $matches)) {
      $index = (int) $matches[1];
      $num_rules = $form_state->get('num_parse_dom_rules');
      $form_state->set('num_parse_dom_rules', $num_rules - 1);

      // Remove the values for this rule.
      $values = $form_state->getValue('parse_dom', []);
      unset($values[$index]);
      $values = array_values($values);
      $form_state->setValue('parse_dom', $values);
    }
    $form_state->setRebuild();
  }

  /**
   * Converts attributes array to string format.
   */
  private function attributesToString($attributes) {
    if (!is_array($attributes)) {
      return '';
    }

    $lines = [];
    foreach ($attributes as $name => $value) {
      $lines[] = $name . '=' . $value;
    }
    return implode("\n", $lines);
  }

  /**
   * Converts string format to attributes array.
   */
  private function stringToAttributes($string) {
    $attributes = [];
    $lines = array_filter(array_map('trim', explode("\n", $string)));
    foreach ($lines as $line) {
      if (strpos($line, '=') !== FALSE) {
        [$name, $value] = explode('=', $line, 2);
        $attributes[trim($name)] = trim($value);
      }
    }
    return $attributes;
  }

  /**
   * Converts string format to variants array.
   */
  private function stringToVariants($string) {
    $variants = [];
    $lines = array_filter(array_map('trim', explode("\n", $string)));

    foreach ($lines as $line) {
      if (strpos($line, '|') !== FALSE) {
        [$key, $label] = explode('|', $line, 2);
        $variants[trim($key)] = trim($label);
      }
    }

    return $variants;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate parse DOM rules.
    $parse_dom = $form_state->getValue('parse_dom', []);
    foreach ($parse_dom as $i => $rule) {
      if (!is_array($rule) || !isset($rule['type'])) {
        continue;
      }

      if ($rule['type'] === 'tag' && empty($rule['tag'])) {
        $form_state->setErrorByName("parse_dom][$i][tag", $this->t('Tag name is required for tag rules.'));
      }
      elseif ($rule['type'] === 'style' && empty($rule['style'])) {
        $form_state->setErrorByName("parse_dom][$i][style", $this->t('Style rule is required for style rules.'));
      }
      elseif ($rule['type'] === 'clearMark' && empty($rule['clear_mark'])) {
        $form_state->setErrorByName("parse_dom][$i][clear_mark", $this->t('Mark type is required for clear mark rules.'));
      }
    }

    // Validate to DOM specification.
    $to_dom = $form_state->getValue('to_dom', []);
    if (empty($to_dom['tag'])) {
      $form_state->setErrorByName('to_dom][tag', $this->t('Tag name is required for DOM output.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\prosemirror\ProseMirrorMarkInterface $mark */
    $mark = $this->entity;

    // Process parse DOM rules.
    $parse_dom = [];
    $raw_rules = $form_state->getValue('parse_dom', []);
    foreach ($raw_rules as $rule) {
      if (!is_array($rule) || !isset($rule['type'])) {
        continue;
      }

      $processed_rule = [];

      if ($rule['type'] === 'tag' && !empty($rule['tag'])) {
        $processed_rule['tag'] = $rule['tag'];
        if (!empty($rule['get_attrs'])) {
          $processed_rule['get_attrs'] = $rule['get_attrs'];
        }
      }
      elseif ($rule['type'] === 'style' && !empty($rule['style'])) {
        $processed_rule['style'] = $rule['style'];
      }
      elseif ($rule['type'] === 'clearMark' && !empty($rule['clear_mark'])) {
        $processed_rule['style'] = $rule['style'] ?? '';
        $processed_rule['clear_mark'] = $rule['clear_mark'];
      }

      if (!empty($processed_rule)) {
        $parse_dom[] = $processed_rule;
      }
    }

    $mark->set('parse_dom', $parse_dom);

    // Process to DOM specification.
    $to_dom = $form_state->getValue('to_dom', []);
    $to_dom_spec = [
      'tag' => $to_dom['tag'],
    ];

    if (!empty($to_dom['attrs'])) {
      $to_dom_spec['attrs'] = $this->stringToAttributes($to_dom['attrs']);
    }

    $mark->set('to_dom', $to_dom_spec);

    // Process additional attributes.
    $attributes = $form_state->getValue('attributes', []);
    $processed_attrs = array_filter($attributes);
    $mark->set('attributes', $processed_attrs);

    // Process variants (for any mark that has variants data)
    $variants_data = $form_state->getValue('variants', []);
    if (!empty($variants_data['variants_text'])) {
      $processed_variants = $this->stringToVariants($variants_data['variants_text']);
      $mark->set('variants', $processed_variants);
    }

    $status = $mark->save();

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addMessage($this->t('Created the %label mark.', [
          '%label' => $mark->label(),
        ]));
        break;

      default:
        $this->messenger()->addMessage($this->t('Saved the %label mark.', [
          '%label' => $mark->label(),
        ]));
    }

    $form_state->setRedirectUrl($mark->toUrl('collection'));
  }

}
