<?php

namespace Drupal\prosemirror\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure ProseMirror settings.
 */
class ProseMirrorSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a ProseMirrorSettingsForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'prosemirror_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['prosemirror.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('prosemirror.settings');

    // Get all available groups from configured elements.
    $groups = $this->getAvailableGroups();

    $form['root_group'] = [
      '#type' => 'select',
      '#title' => $this->t('Document Root Group'),
      '#description' => $this->t('Select which group of elements can be direct children of the document root.'),
      '#options' => $groups,
      '#default_value' => $config->get('root_group') ?? 'block',
      '#required' => TRUE,
    ];

    $form['text_group'] = [
      '#type' => 'select',
      '#title' => $this->t('Text Node Group'),
      '#description' => $this->t('Select which group the text node belongs to. This determines where text content can appear.'),
      '#options' => $groups,
      '#default_value' => $config->get('text_group') ?? 'inline',
      '#required' => TRUE,
    ];

    // Get block elements for default element selection.
    $block_elements = $this->getBlockElements();

    $form['default_element'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Element'),
      '#description' => $this->t('Select which element type should be created when the editor is empty. Best practice is to use "paragraph" as the default element type, as it provides a neutral starting point for content creation.'),
      '#options' => $block_elements,
      '#default_value' => $config->get('default_element') ?? 'paragraph',
      '#required' => TRUE,
    ];

    // Code block settings.
    $form['code_blocks'] = [
      '#type' => 'details',
      '#title' => $this->t('Code Blocks'),
      '#description' => $this->t('Configure settings for code block elements.'),
      '#open' => FALSE,
    ];

    $form['code_blocks']['code_block_languages'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Available Languages'),
      '#default_value' => $this->languagesToString($config->get('code_block_languages') ?? _prosemirror_get_default_code_block_languages()),
      '#description' => $this->t('Enter one language per line in the format: <code>key|label</code><br><strong>Examples:</strong><br><code>javascript|JavaScript</code><br><code>php|PHP</code><br><code>plaintext|Plain text</code>'),
      '#rows' => 10,
    ];

    // Table settings.
    $form['tables'] = [
      '#type' => 'details',
      '#title' => $this->t('Tables'),
      '#description' => $this->t('Configure settings for table elements.'),
      '#open' => FALSE,
    ];

    $form['tables']['table_max_rows'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Rows'),
      '#default_value' => $config->get('table_max_rows') ?? 50,
      '#min' => 1,
      '#max' => 100,
      '#description' => $this->t('Maximum number of rows allowed in tables.'),
    ];

    $form['tables']['table_max_columns'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Columns'),
      '#default_value' => $config->get('table_max_columns') ?? 10,
      '#min' => 1,
      '#max' => 20,
      '#description' => $this->t('Maximum number of columns allowed in tables.'),
    ];

    $form['tables']['table_allow_header_rows'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow Header Rows'),
      '#default_value' => $config->get('table_allow_header_rows') ?? TRUE,
      '#description' => $this->t('Allow tables to have header rows.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Converts languages array to string format for textarea.
   */
  private function languagesToString($languages): string {
    if (!is_array($languages)) {
      return '';
    }
    $lines = [];
    foreach ($languages as $key => $label) {
      $lines[] = $key . '|' . $label;
    }
    return implode("\n", $lines);
  }

  /**
   * Converts string format back to languages array.
   */
  private function stringToLanguages($string): array {
    $languages = [];
    $lines = array_filter(array_map('trim', explode("\n", $string)));
    foreach ($lines as $line) {
      if (strpos($line, '|') !== FALSE) {
        [$key, $label] = explode('|', $line, 2);
        $languages[trim($key)] = trim($label);
      }
    }
    return $languages;
  }

  /**
   * Gets all available groups from configured elements.
   *
   * @return array
   *   An array of group options.
   */
  protected function getAvailableGroups() {
    $groups = [];

    // Load all element groups.
    $group_storage = $this->entityTypeManager->getStorage('prosemirror_element_group');
    $element_groups = $group_storage->loadMultiple();

    foreach ($element_groups as $group) {
      $groups[$group->id()] = $group->label();
    }

    // Also get groups from elements that might not have a group entity.
    $element_storage = $this->entityTypeManager->getStorage('prosemirror_element');
    $elements = $element_storage->loadMultiple();

    foreach ($elements as $element) {
      $element_groups = $element->getGroups();
      foreach ($element_groups as $group_id) {
        // Add groups that don't have a group entity yet.
        if (!isset($groups[$group_id])) {
          $groups[$group_id] = ucfirst($group_id);
        }
      }
    }

    asort($groups);

    return $groups;
  }

  /**
   * Gets all block elements that can be used as default.
   *
   * @return array
   *   An array of element options.
   */
  protected function getBlockElements() {
    $elements = [];

    // Load all elements.
    $element_storage = $this->entityTypeManager->getStorage('prosemirror_element');
    $all_elements = $element_storage->loadMultiple();

    // Get the configured root group.
    $config = $this->config('prosemirror.settings');
    $root_group = $config->get('root_group') ?? 'block';

    foreach ($all_elements as $element) {
      $groups = $element->getGroups();
      // Include elements that belong to the root group or any block-related group.
      foreach ($groups as $group) {
        if ($group === $root_group || strpos($group, 'block') !== FALSE) {
          $elements[$element->id()] = $element->label();
          break;
        }
      }
    }

    asort($elements);

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $code_blocks = $form_state->getValue('code_blocks');
    $tables = $form_state->getValue('tables');

    $this->config('prosemirror.settings')
      ->set('root_group', $form_state->getValue('root_group'))
      ->set('text_group', $form_state->getValue('text_group'))
      ->set('default_element', $form_state->getValue('default_element'))
      ->set('code_block_languages', $this->stringToLanguages($code_blocks['code_block_languages'] ?? ''))
      ->set('table_max_rows', $tables['table_max_rows'] ?? 50)
      ->set('table_max_columns', $tables['table_max_columns'] ?? 10)
      ->set('table_allow_header_rows', $tables['table_allow_header_rows'] ?? TRUE)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
