<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\ElementType;

use Drupal\prosemirror\Plugin\ProseMirrorElementTypeBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\prosemirror\ProseMirrorElementInterface;
use Drupal\prosemirror\Transformation\ValidationError;

/**
 * Provides a 'base_node' element type for basic ProseMirror nodes.
 *
 * @ProseMirrorElementType(
 *   id = "base_node",
 *   label = @Translation("Base Node"),
 *   description = @Translation("A basic node type for standard HTML elements like paragraphs, headings, etc.")
 * )
 */
class BaseNode extends ProseMirrorElementTypeBase {

  /**
   * {@inheritdoc}
   */
  public function getAvailableOptions() {
    return [
      'tag' => [
        'label' => $this->t('HTML Tag'),
        'type' => 'string',
        'default' => 'p',
        'required' => TRUE,
        'description' => $this->t('The HTML tag to use for this node (e.g., p, blockquote, h1-h6, hr, br).'),
      ],
      // See https://prosemirror.net/docs/ref/#model.NodeSpec.inline for more information.
      'inline' => [
        'label' => $this->t('Inline'),
        'type' => 'boolean',
        'default' => FALSE,
        'required' => FALSE,
        'description' => $this->t('Whether this is an inline node (like br) or a block node.'),
      ],
      // See https://prosemirror.net/docs/ref/#model.NodeSpec.defining for more information.
      'defining' => [
        'label' => $this->t('Defining'),
        'type' => 'boolean',
        'default' => FALSE,
        'required' => FALSE,
        'description' => $this->t('Whether content inside this node is isolated from the outside.'),
      ],
      // See https://prosemirror.net/docs/ref/#model.NodeSpec.selectable for more information.
      'selectable' => [
        'label' => $this->t('Selectable'),
        'type' => 'boolean',
        'default' => TRUE,
        'required' => FALSE,
        'description' => $this->t('Whether this node can be selected as a node selection.'),
      ],
      // Special behavior for code blocks.
      'code' => [
        'label' => $this->t('Code'),
        'type' => 'boolean',
        'default' => FALSE,
        'required' => FALSE,
        'description' => $this->t('Whether this node contains code content.'),
      ],
      // See https://prosemirror.net/docs/ref/#model.NodeSpec.marks for more information.
      'marks' => [
        'label' => $this->t('Allowed Marks'),
        'type' => 'string',
        'default' => '_',
        'required' => FALSE,
        'description' => $this->t('Which marks are allowed inside this node. Use "_" for all marks, "" for no marks, or a space-separated list of mark names.'),
      ],
      // See https://prosemirror.net/docs/ref/#model.NodeSpec.attrs for more information.
      'attrs' => [
        'label' => $this->t('Attributes'),
        'type' => 'array',
        'default' => [],
        'required' => FALSE,
        'description' => $this->t('Additional attributes for this node type.'),
      ],
      // See https://prosemirror.net/docs/ref/#model.NodeSpec.parseDOM for more information.
      'parseDOM' => [
        'label' => $this->t('Parse DOM Rules'),
        'type' => 'array',
        'default' => [],
        'required' => FALSE,
        'description' => $this->t('Additional parse rules for this node.'),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(array $form, FormStateInterface $form_state, ProseMirrorElementInterface $element) {
    $options = $element->getOptions();

    // HTML tag field.
    $form['tag'] = [
      '#type' => 'textfield',
      '#title' => $this->t('HTML Tag'),
      '#default_value' => $options['tag'] ?? 'p',
      '#required' => TRUE,
      '#description' => $this->t('Enter a valid HTML tag name (e.g., p, div, section, blockquote, h1-h6, hr, br, pre, article, aside, nav, header, footer, main, etc.).'),
      '#attributes' => [
        'placeholder' => 'p',
      ],
      '#size' => 20,
      '#maxlength' => 50,
    ];

    // Inline field.
    $form['inline'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Inline'),
      '#default_value' => $options['inline'] ?? FALSE,
      '#description' => $this->t('Check if this is an inline node (like br).'),
    ];

    // Defining field.
    $form['defining'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Defining'),
      '#default_value' => $options['defining'] ?? FALSE,
      '#description' => $this->t('Check if content inside this node is isolated from the outside.'),
    ];

    // Selectable field.
    $form['selectable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Selectable'),
      '#default_value' => $options['selectable'] ?? TRUE,
      '#description' => $this->t('Whether this node can be selected as a node selection.'),
    ];

    // Marks field.
    $form['marks'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Allowed Marks'),
      '#default_value' => $options['marks'] ?? '_',
      '#description' => $this->t('Use "_" for all marks, "" for no marks, or space-separated mark names.'),
    ];

    // Attributes field for headings.
    $form['level'] = [
      '#type' => 'value',
      '#value' => NULL,
    ];

    // Parse DOM rules.
    $form['parseDOM_extra'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional Parse DOM Rules'),
      '#default_value' => $options['parseDOM_extra'] ?? '',
      '#description' => $this->t('Additional CSS selectors for parsing, one per line (e.g., "i" for italic tags).'),
      '#rows' => 3,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('type_options');

    // Validate HTML tag.
    if (!empty($values['tag'])) {
      $tag = strtolower(trim($values['tag']));

      // Validate tag name format (letters, numbers, no spaces)
      if (!preg_match('/^[a-z][a-z0-9]*$/', $tag)) {
        $form_state->setErrorByName('type_options][tag', $this->t('HTML tag must start with a letter and contain only lowercase letters and numbers.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function processFormValues(array $values) {
    // Normalize tag to lowercase.
    if (isset($values['tag'])) {
      $values['tag'] = strtolower(trim($values['tag']));
    }

    $values['inline'] = $values['inline'] ?? FALSE;
    $values['defining'] = $values['defining'] ?? FALSE;
    $values['selectable'] = $values['selectable'] ?? TRUE;
    $values['code'] = $values['code'] ?? FALSE;
    $values['marks'] = $values['marks'] ?? '_';
    $values['attrs'] = $values['attrs'] ?? [];

    // Set inline based on tag.
    if ($values['tag'] === 'br') {
      $values['inline'] = TRUE;
      $values['selectable'] = FALSE;
    }
    elseif ($values['tag'] === 'hr') {
      $values['marks'] = '';
    }

    // Set level attribute for headings.
    if (preg_match('/^h(\d)$/', $values['tag'], $matches)) {
      $values['level'] = (int) $matches[1];
      $values['defining'] = TRUE;
    }
    else {
      unset($values['level']);
    }

    // Process parseDOM_extra.
    $values['parseDOM'] = [];
    if (!empty($values['parseDOM_extra'])) {
      $lines = array_filter(array_map('trim', explode("\n", $values['parseDOM_extra'])));
      foreach ($lines as $line) {
        $values['parseDOM'][] = ['tag' => $line];
      }
    }
    unset($values['parseDOM_extra']);

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function buildJavaScriptConfiguration(ProseMirrorElementInterface $element) {
    $config = parent::buildJavaScriptConfiguration($element);

    // Add baseNode-specific configuration.
    $options = $element->getOptions();
    $config['options']['nodeType'] = 'base';
    $config['options']['tag'] = $options['tag'] ?? 'p';
    $config['options']['inline'] = $options['inline'] ?? FALSE;
    $config['options']['defining'] = $options['defining'] ?? FALSE;
    $config['options']['selectable'] = $options['selectable'] ?? TRUE;
    $config['options']['code'] = $options['code'] ?? FALSE;
    $config['options']['marks'] = $options['marks'] ?? '_';

    // Add level attribute for headings.
    if (isset($options['level'])) {
      $config['options']['attrs'] = ['level' => ['default' => $options['level'], 'validate' => 'number']];
    }

    // Add parse DOM rules.
    $parseDOM = [[
      'tag' => $config['options']['tag'],
    ]
];
    if (!empty($options['parseDOM'])) {
      $parseDOM = array_merge($parseDOM, $options['parseDOM']);
    }
    $config['options']['parseDOM'] = $parseDOM;

    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function validateNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    $sanitized = [
      'type' => $node['type'],
    ];

    // Validate attributes - BaseNode strips all attributes.
    if (isset($node['attrs']) && is_array($node['attrs'])) {
      $sanitizedAttrs = $this->validateAttributes($node['attrs'], $path, $errors, $references);
      // Only include attrs if validateAttributes returns non-empty array.
      if (!empty($sanitizedAttrs)) {
        $sanitized['attrs'] = $sanitizedAttrs;
      }
    }

    // Validate marks.
    if (isset($node['marks']) && is_array($node['marks']) && count($node['marks'])) {
      // Let the transformation helper handle marks validation.
      $sanitized['marks'] = $transformationHelper->sanitizeMarks($node['marks'], $path, $errors, $references);
    }

    // Validate content.
    if (isset($node['content']) && is_array($node['content']) && count($node['content'])) {
      $sanitized['content'] = [];
      foreach ($node['content'] as $index => $child) {
        $childPath = array_merge($path, ['content', $index]);
        $sanitizedChild = $transformationHelper->validateChildNode($child, $childPath, $errors, $references);
        if (!empty($sanitizedChild)) {
          $sanitized['content'][] = $sanitizedChild;
        }
      }
    }

    // Handle text nodes.
    if (isset($node['text'])) {
      $sanitized['text'] = (string) $node['text'];
    }

    return $sanitized;
  }

  /**
   * {@inheritdoc}
   */
  public function validateAttributes(array $attrs, array $path, array &$errors, array &$references): array {
    return [];
  }

}
