<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\ElementType;

use Drupal\prosemirror\Plugin\ProseMirrorElementTypeBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\prosemirror\ProseMirrorElementInterface;
use Drupal\prosemirror\Transformation\ValidationError;

/**
 * Provides a 'list_block' element type.
 *
 * @ProseMirrorElementType(
 *   id = "list_block",
 *   label = @Translation("List Block"),
 *   description = @Translation("A block element that contains a list of other elements.")
 * )
 */
class ListBlock extends ProseMirrorElementTypeBase {

  /**
   * {@inheritdoc}
   */
  public function getAvailableOptions() {
    return [
      // Whether to render the list as a column or a row.
      'direction' => [
        'label' => $this->t('Direction'),
        'type' => 'string',
        'default' => 'column',
        'required' => FALSE,
        'options' => [
          'column' => $this->t('Column'),
          'row' => $this->t('Row'),
        ],
      ],
      // Maximum number of child elements allowed.
      'maxChildren' => [
        'label' => $this->t('Maximum Children'),
        'type' => 'integer',
        'default' => NULL,
        'required' => FALSE,
      ],
      // The element type to use for child blocks. Can be either a group or an element.
      'contentBlock' => [
        'label' => $this->t('Content Block'),
        'type' => 'string',
        'default' => '',
        'required' => FALSE,
        'description' => $this->t('The element type to use for child blocks.'),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(array $form, FormStateInterface $form_state, ProseMirrorElementInterface $element) {
    $options = $element->getOptions();

    // Direction field.
    $form['direction'] = [
      '#type' => 'select',
      '#title' => $this->t('Direction'),
      '#options' => [
        'column' => $this->t('Column'),
        'row' => $this->t('Row'),
      ],
      '#default_value' => $options['direction'] ?? 'column',
      '#description' => $this->t('The direction of the list layout.'),
    ];

    // Maximum children field.
    $form['maxChildren'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Children'),
      '#default_value' => $options['maxChildren'] ?? NULL,
      '#min' => 1,
      '#max' => 100,
      '#description' => $this->t('The maximum number of child elements allowed.'),
    ];

    // Content block selection.
    $content_options = $this->getContentBlockOptions();
    $form['contentBlock'] = [
      '#type' => 'select',
      '#title' => $this->t('Content Block'),
      '#options' => $content_options,
      '#default_value' => $options['contentBlock'] ?? '',
      '#empty_option' => $this->t('- None -'),
      '#description' => $this->t('The element type or group to use for child blocks.'),
    ];

    return $form;
  }

  /**
   * Gets the available content block options.
   */
  protected function getContentBlockOptions() {
    $options = [];

    // Add ProseMirror defaults.
    $options[$this->t('ProseMirror Defaults')->__toString()] = [
      'paragraph' => $this->t('Paragraph'),
      'inline' => $this->t('Inline'),
      'text' => $this->t('Text'),
    ];

    // Add element groups.
    $group_storage = \Drupal::entityTypeManager()->getStorage('prosemirror_element_group');
    $groups = $group_storage->loadMultiple();
    if (!empty($groups)) {
      $group_options = [];
      foreach ($groups as $group) {
        $group_options[$group->id()] = $group->label();
      }
      $options[$this->t('Element Groups')->__toString()] = $group_options;
    }

    // Add elements.
    $element_storage = \Drupal::entityTypeManager()->getStorage('prosemirror_element');
    $elements = $element_storage->loadMultiple();
    if (!empty($elements)) {
      $element_options = [];
      foreach ($elements as $element) {
        $element_options[$element->id()] = $element->label();
      }
      $options[$this->t('Elements')->__toString()] = $element_options;
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('type_options');

    // Validate maxChildren.
    if (!empty($values['maxChildren'])) {
      $max = $values['maxChildren'];
      if ($max < 1) {
        $form_state->setErrorByName('type_options][maxChildren', $this->t('Maximum children must be at least 1.'));
      }
      elseif ($max > 100) {
        $form_state->setErrorByName('type_options][maxChildren', $this->t('Maximum children cannot exceed 100.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function processFormValues(array $values) {
    // Convert empty maxChildren to NULL.
    if (isset($values['maxChildren']) && $values['maxChildren'] === '') {
      $values['maxChildren'] = NULL;
    }
    elseif (isset($values['maxChildren'])) {
      $values['maxChildren'] = (int) $values['maxChildren'];
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptions(array $options) {
    $errors = parent::validateOptions($options);

    // Validate direction.
    if (!empty($options['direction'])) {
      $valid_directions = ['column', 'row'];
      if (!in_array($options['direction'], $valid_directions, TRUE)) {
        $errors['direction'] = $this->t('Direction must be either "column" or "row".');
      }
    }

    // Validate maxChildren.
    if (isset($options['maxChildren'])) {
      if (!is_numeric($options['maxChildren']) || $options['maxChildren'] < 1) {
        $errors['maxChildren'] = $this->t('Maximum children must be a positive number.');
      }
    }

    // Validate contentBlock.
    if (!empty($options['contentBlock'])) {
      // Check if it's a valid machine name pattern.
      if (!preg_match('/^[a-z0-9_]+$/', $options['contentBlock'])) {
        $errors['contentBlock'] = $this->t('Content block must be a valid machine name (lowercase letters, numbers, and underscores only).');
      }
    }

    return $errors;
  }

  /**
   * {@inheritdoc}
   */
  public function validateNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    $sanitized = parent::validateNode($node, $path, $errors, $references, $transformationHelper);

    $options = $this->getOptionsConfig();

    // Validate maxChildren constraint.
    if (isset($options['maxChildren']) && $options['maxChildren'] > 0) {
      if (isset($sanitized['content']) && is_array($sanitized['content'])) {
        $childCount = count($sanitized['content']);
        if ($childCount > $options['maxChildren']) {
          $errors[] = ValidationError::atPath("Too many children: found $childCount, maximum allowed is {$options['maxChildren']}.", $path);

          // Truncate content to maxChildren limit.
          $sanitized['content'] = array_slice($sanitized['content'], 0, $options['maxChildren']);
        }
      }
    }

    // Validate contentBlock constraint.
    if (!empty($options['contentBlock']) && isset($sanitized['content']) && is_array($sanitized['content']) && count($sanitized['content']) > 0) {
      $allowedTypes = $this->getAllowedContentTypes($options['contentBlock']);

      if (!empty($allowedTypes)) {
        $validContent = [];
        foreach ($sanitized['content'] as $index => $child) {
          if (isset($child['type'])) {
            $childType = $child['type'];
            if (in_array($childType, $allowedTypes, TRUE)) {
              $validContent[] = $child;
            }
            else {
              $childPath = array_merge($path, ['content', $index]);
              $errors[] = ValidationError::atPath("Invalid child type '$childType'. Allowed types for '{$options['contentBlock']}' are: " . implode(', ', $allowedTypes), $childPath);
            }
          }
        }
        $sanitized['content'] = $validContent;
      }
    }

    return $sanitized;
  }

  /**
   * Gets the allowed content types for a given contentBlock setting.
   *
   * @param string $contentBlock
   *   The contentBlock setting (element ID or group ID).
   *
   * @return array
   *   Array of allowed element type IDs.
   */
  protected function getAllowedContentTypes($contentBlock) {
    $allowedTypes = [];

    // Handle ProseMirror defaults.
    if (in_array($contentBlock, ['paragraph', 'inline', 'text'], TRUE)) {
      return [$contentBlock];
    }

    // Check if it's a single element reference.
    $element_storage = \Drupal::entityTypeManager()->getStorage('prosemirror_element');
    $element = $element_storage->load($contentBlock);
    if ($element) {
      return [$contentBlock];
    }

    // Check if it's a group reference.
    $group_storage = \Drupal::entityTypeManager()->getStorage('prosemirror_element_group');
    $group = $group_storage->load($contentBlock);
    if ($group) {
      // Get all elements that belong to this group.
      $elements = $group->getElements();
      foreach ($elements as $element) {
        $allowedTypes[] = $element->id();
      }
    }

    return $allowedTypes;
  }

}
