<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\Rendering;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\prosemirror\Rendering\Plugin\ProseMirrorRenderingPluginBase;
use Drupal\prosemirror\Rendering\ProseMirrorRenderer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Renders markup elements like text with marks (bold, italic, links, etc.).
 *
 * @ProseMirrorRenderingPlugin(
 *   id = "markup",
 *   label = @Translation("Markup Renderer"),
 *   description = @Translation("Renders text nodes with marks like bold, italic, links, etc."),
 *   node_types = {"text"},
 *   formats = {"html"},
 *   weight = 50
 * )
 */
class MarkupRenderer extends ProseMirrorRenderingPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a MarkupRenderer.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    if (!$this->supportsFormat($format) || $node['type'] !== 'text') {
      return FALSE;
    }

    $text = $node['text'] ?? '';
    $marks = $node['marks'] ?? [];

    // If no marks, just return plain text (safely escaped)
    if (empty($marks)) {
      $renderArray = ['#plain_text' => $text];
      return TRUE;
    }

    // Apply marks to the text.
    $renderArray = $this->applyMarksToText($text, $marks);

    return TRUE;
  }

  /**
   * Applies marks to text content.
   *
   * @param string $text
   *   The text content.
   * @param array $marks
   *   Array of marks to apply.
   *
   * @return array
   *   The render array with marks applied.
   */
  protected function applyMarksToText(string $text, array $marks): array {
    // Start with the text (safely escaped)
    $renderArray = ['#plain_text' => $text];

    // Apply marks from innermost to outermost.
    foreach ($marks as $mark) {
      $renderArray = $this->wrapWithMark($renderArray, $mark);
    }

    return $renderArray;
  }

  /**
   * Wraps content with a specific mark.
   *
   * @param array $content
   *   The content to wrap.
   * @param array $mark
   *   The mark definition.
   *
   * @return array
   *   The wrapped render array.
   */
  protected function wrapWithMark(array $content, array $mark): array {
    $type = $mark['type'] ?? '';
    $attrs = $mark['attrs'] ?? [];

    switch ($type) {
      case 'bold':
      case 'strong':
        return [
          '#type' => 'html_tag',
          '#tag' => 'strong',
          'content' => $content,
        ];

      case 'italic':
      case 'em':
        return [
          '#type' => 'html_tag',
          '#tag' => 'em',
          'content' => $content,
        ];

      case 'underline':
        return [
          '#type' => 'html_tag',
          '#tag' => 'u',
          'content' => $content,
        ];

      case 'strike':
      case 'strikethrough':
        return [
          '#type' => 'html_tag',
          '#tag' => 's',
          'content' => $content,
        ];

      case 'code':
        return [
          '#type' => 'html_tag',
          '#tag' => 'code',
          'content' => $content,
        ];

      case 'small':
        return [
          '#type' => 'html_tag',
          '#tag' => 'small',
          'content' => $content,
        ];

      case 'subscript':
        return [
          '#type' => 'html_tag',
          '#tag' => 'sub',
          'content' => $content,
        ];

      case 'superscript':
        return [
          '#type' => 'html_tag',
          '#tag' => 'sup',
          'content' => $content,
        ];

      case 'link':
        return $this->renderLink($content, $attrs);

      default:
        // For unknown marks, just return the content unchanged.
        return $content;
    }
  }

  /**
   * Renders a link using Drupal's link system.
   *
   * @param array $content
   *   The link content.
   * @param array $attrs
   *   The link attributes.
   *
   * @return array
   *   The link render array.
   */
  protected function renderLink(array $content, array $attrs): array {
    $href = $attrs['href'] ?? '';
    $title = $attrs['title'] ?? '';
    $linkType = $attrs['linkType'] ?? 'external';
    $entityType = $attrs['entityType'] ?? '';
    $entityUuid = $attrs['entityUuid'] ?? '';

    if (empty($href) && empty($entityUuid)) {
      // If no href or entity reference, just return the content.
      return $content;
    }

    try {
      $url = NULL;

      // Handle entity references first.
      if (!empty($entityUuid) && !empty($entityType)) {
        $url = $this->createEntityUrl($entityType, $entityUuid);
        if (!$url) {
          // If entity URL creation failed, return content as plain text.
          return $content;
        }
      }
      // Handle regular links.
      elseif (!empty($href)) {
        if ($linkType === 'internal') {
          // Internal Drupal link.
          $url = Url::fromUserInput($href);
        }
        else {
          // External link.
          $url = Url::fromUri($href);
        }
      }

      if (!$url) {
        return $content;
      }

      $linkOptions = [];
      if (!empty($title)) {
        $linkOptions['attributes']['title'] = $title;
      }

      // Add target="_blank" for external links.
      if ($linkType === 'external' || (!empty($entityType) && $entityType !== 'node')) {
        $linkOptions['attributes']['target'] = '_blank';
        $linkOptions['attributes']['rel'] = 'noopener noreferrer';
      }

      $url->setOptions($linkOptions);

      return [
        '#type' => 'link',
        '#url' => $url,
        '#title' => $content,
      ];
    }
    catch (\Exception $e) {
      // If URL creation fails, render as plain text with the original content.
      \Drupal::logger('prosemirror')->warning('Failed to create URL for link: @href. Error: @error', [
        '@href' => $href,
        '@error' => $e->getMessage(),
      ]);

      return $content;
    }
  }

  /**
   * Creates a URL for an entity reference.
   *
   * @param string $entityType
   *   The entity type.
   * @param string $entityUuid
   *   The entity UUID.
   *
   * @return \Drupal\Core\Url|null
   *   The entity URL or NULL if not available.
   */
  protected function createEntityUrl(string $entityType, string $entityUuid): ?Url {
    try {
      // Check if the entity type exists and has a canonical link template.
      $entityTypeDefinition = $this->entityTypeManager->getDefinition($entityType);
      if (!$entityTypeDefinition || !$entityTypeDefinition->hasLinkTemplate('canonical')) {
        return NULL;
      }

      // Load the entity by UUID.
      $storage = $this->entityTypeManager->getStorage($entityType);
      $entities = $storage->loadByProperties(['uuid' => $entityUuid]);

      if (empty($entities)) {
        return NULL;
      }

      $entity = reset($entities);

      // Check if entity is published (if it has a status field)
      if ($entity->hasField('status') && !$entity->isPublished()) {
        return NULL;
      }

      // Create the canonical URL.
      return $entity->toUrl('canonical');
    }
    catch (\Exception $e) {
      \Drupal::logger('prosemirror')->warning('Failed to create entity URL for @type:@uuid. Error: @error', [
        '@type' => $entityType,
        '@uuid' => $entityUuid,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
