<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\Rendering;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\prosemirror\Rendering\Plugin\ProseMirrorRenderingPluginBase;
use Drupal\prosemirror\Rendering\ProseMirrorRenderer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Renders media elements.
 *
 * @ProseMirrorRenderingPlugin(
 *   id = "media",
 *   label = @Translation("Media Renderer"),
 *   description = @Translation("Renders media elements."),
 *   node_types = {"media"},
 *   formats = {"html"},
 *   weight = 25
 * )
 */
class MediaRenderer extends ProseMirrorRenderingPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a MediaRenderer.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    if (!$this->supportsFormat($format)) {
      return FALSE;
    }

    $type = $node['type'];

    switch ($type) {
      case 'media':
        return $this->renderMedia($node, $renderArray);
    }

    return FALSE;
  }

  /**
   * Renders a media element.
   *
   * @param array $node
   *   The media node.
   * @param array &$renderArray
   *   The render array to populate.
   *
   * @return bool
   *   TRUE if rendered successfully.
   */
  protected function renderMedia(array $node, array &$renderArray): bool {
    $attrs = $node['attrs'] ?? [];
    $entityType = $attrs['data-entity-type'] ?? 'media';
    $entityUuid = $attrs['data-entity-uuid'] ?? '';

    if (empty($entityUuid)) {
      return FALSE;
    }

    try {
      $storage = $this->entityTypeManager->getStorage($entityType);
      $entities = $storage->loadByProperties(['uuid' => $entityUuid]);

      if (empty($entities)) {
        return FALSE;
      }

      $entity = reset($entities);

      // Use Drupal's entity view builder.
      $viewBuilder = $this->entityTypeManager->getViewBuilder($entityType);
      $renderArray = $viewBuilder->view($entity, 'default');

      // Add ProseMirror-specific classes.
      $renderArray['#attributes']['class'][] = 'pm-media';

      return TRUE;
    }
    catch (\Exception $e) {
      \Drupal::logger('prosemirror')->error('Error rendering media: @error', ['@error' => $e->getMessage()]);
      return FALSE;
    }
  }

}
