<?php

namespace Drupal\prosemirror\Plugin;

use Drupal\Core\Plugin\PluginBase;
use Drupal\prosemirror\ProseMirrorElementInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\prosemirror\Transformation\ValidationError;

/**
 * Base class for ProseMirror element type plugins.
 */
abstract class ProseMirrorElementTypeBase extends PluginBase implements ProseMirrorElementTypeInterface {

  /**
   * Gets the options configuration.
   *
   * @return array
   *   The options configuration.
   */
  protected function getOptionsConfig() {
    return $this->configuration ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptions(array $options) {
    $errors = [];
    $available_options = $this->getAvailableOptions();

    foreach ($available_options as $option_name => $option_def) {
      if (!empty($option_def['required']) && empty($options[$option_name])) {
        $errors[$option_name] = t('@option is required.', ['@option' => $option_def['label']]);
      }
    }

    return $errors;
  }

  /**
   * {@inheritdoc}
   */
  public function buildJavaScriptConfiguration(ProseMirrorElementInterface $element) {
    $content = $element->getContent();
    $expression = $element->getContentExpression();

    // Build the full content expression.
    $contentExpression = '';
    if ($content) {
      $contentExpression = $content . $expression;
    }

    $config = [
      'type' => $this->getPluginId(),
      'options' => array_merge(
        [
          'name' => $element->id(),
          'displayName' => $element->label(),
          'classes' => $element->getCssClasses(),
          'content' => $contentExpression,
          'group' => implode(' ', $element->getGroups()),
        ],
        $element->getOptions()
      ),
    ];

    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(array $form, FormStateInterface $form_state, ProseMirrorElementInterface $element) {
    // Default implementation - subclasses should override this.
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(array &$form, FormStateInterface $form_state) {
    // Default implementation - subclasses should override this.
  }

  /**
   * {@inheritdoc}
   */
  public function processFormValues(array $values) {
    // Default implementation - return values as-is.
    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function validateNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    $sanitized = [
      'type' => $node['type'],
    ];

    // Validate attributes.
    if (isset($node['attrs']) && is_array($node['attrs'])) {
      $sanitized['attrs'] = $this->validateAttributes($node['attrs'], $path, $errors, $references);
    }

    // Validate marks.
    if (isset($node['marks']) && is_array($node['marks']) && count($node['marks'])) {
      // Let the transformation helper handle marks validation.
      $sanitized['marks'] = $transformationHelper->sanitizeMarks($node['marks'], $path, $errors, $references);
    }

    // Validate content.
    if (isset($node['content']) && is_array($node['content']) && count($node['content'])) {
      $sanitized['content'] = [];
      foreach ($node['content'] as $index => $child) {
        $childPath = array_merge($path, ['content', $index]);
        $sanitizedChild = $transformationHelper->validateChildNode($child, $childPath, $errors, $references);
        if (!empty($sanitizedChild)) {
          $sanitized['content'][] = $sanitizedChild;
        }
      }
    }

    // Handle text nodes.
    if (isset($node['text'])) {
      $sanitized['text'] = (string) $node['text'];
    }

    return $sanitized;
  }

  /**
   * {@inheritdoc}
   */
  public function validateAttributes(array $attrs, array $path, array &$errors, array &$references): array {
    // Must be overwritten by subclasses to whitelist and validate attributes.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function enrichAttributesForEditorDisplay(array $attrs, array $node): array {
    // Default implementation - return attributes as-is.
    // Subclasses should override this to add display-specific attributes.
    return $attrs;
  }

}
