<?php

namespace Drupal\prosemirror\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\prosemirror\ProseMirrorElementInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines an interface for ProseMirror element type plugins.
 */
interface ProseMirrorElementTypeInterface extends PluginInspectionInterface {

  /**
   * Returns the element type label.
   *
   * @return string
   *   The element type label.
   */
  public function label();

  /**
   * Returns the available options for this element type.
   *
   * @return array
   *   An array of option definitions keyed by option name.
   *   Each option should have:
   *   - label: The human-readable label for the option.
   *   - type: The data type (string, integer, boolean, array).
   *   - default: The default value.
   *   - required: Whether the option is required.
   */
  public function getAvailableOptions();

  /**
   * Validates the options for this element type.
   *
   * @param array $options
   *   The options to validate.
   *
   * @return array
   *   An array of validation errors, keyed by option name.
   */
  public function validateOptions(array $options);

  /**
   * Builds the configuration for export to JavaScript.
   *
   * @param \Drupal\prosemirror\ProseMirrorElementInterface $element
   *   The element entity.
   *
   * @return array
   *   The configuration array for JavaScript.
   */
  public function buildJavaScriptConfiguration(ProseMirrorElementInterface $element);

  /**
   * Builds the form elements for type-specific options.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\prosemirror\ProseMirrorElementInterface $element
   *   The element being edited.
   *
   * @return array
   *   The form array with added elements.
   */
  public function buildOptionsForm(array $form, FormStateInterface $form_state, ProseMirrorElementInterface $element);

  /**
   * Validates the form elements for type-specific options.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateOptionsForm(array &$form, FormStateInterface $form_state);

  /**
   * Processes the form values for type-specific options.
   *
   * @param array $values
   *   The form values for the options.
   *
   * @return array
   *   The processed options ready to be saved.
   */
  public function processFormValues(array $values);

  /**
   * Validates a complete node of this element type.
   *
   * @param array $node
   *   The node to validate.
   * @param array $path
   *   The path to this node in the document tree.
   * @param array &$errors
   *   Array to collect validation errors.
   * @param array &$references
   *   Array to collect entity references.
   * @param \Drupal\prosemirror\Transformation\TransformationHelper $transformationHelper
   *   The transformation helper for validating nested content.
   *
   * @return array
   *   The sanitized node.
   */
  public function validateNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array;

  /**
   * Validates node attributes for this element type.
   *
   * @param array $attrs
   *   The attributes to validate.
   * @param array $path
   *   The path to this node in the document tree.
   * @param array &$errors
   *   Array to collect validation errors.
   * @param array &$references
   *   Array to collect entity references.
   *
   * @return array
   *   The sanitized attributes.
   */
  public function validateAttributes(array $attrs, array $path, array &$errors, array &$references): array;

  /**
   * Enriches node attributes for editor display.
   *
   * This method is called when the editor loads content to add display-specific
   * attributes that were stripped during validation/serialization, such as
   * entity labels and URLs for links.
   *
   * @param array $attrs
   *   The current attributes.
   * @param array $node
   *   The full node data.
   *
   * @return array
   *   The enriched attributes.
   */
  public function enrichAttributesForEditorDisplay(array $attrs, array $node): array;

}
