<?php

namespace Drupal\prosemirror\Rendering;

use Drupal\Core\Render\RendererInterface;
use Drupal\prosemirror\Rendering\Plugin\ProseMirrorRenderingPluginManager;
use Drupal\prosemirror\Transformation\TransformationHelper;

/**
 * Service for rendering ProseMirror content.
 */
class ProseMirrorRenderer implements ProseMirrorRendererInterface {

  /**
   * The transformation helper.
   *
   * @var \Drupal\prosemirror\Transformation\TransformationHelper
   */
  protected TransformationHelper $transformationHelper;

  /**
   * The rendering plugin manager.
   *
   * @var \Drupal\prosemirror\Rendering\Plugin\ProseMirrorRenderingPluginManager
   */
  protected ProseMirrorRenderingPluginManager $pluginManager;

  /**
   * The Drupal renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * Constructs a ProseMirrorRenderer.
   *
   * @param \Drupal\prosemirror\Transformation\TransformationHelper $transformation_helper
   *   The transformation helper.
   * @param \Drupal\prosemirror\Rendering\Plugin\ProseMirrorRenderingPluginManager $plugin_manager
   *   The rendering plugin manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The Drupal renderer.
   */
  public function __construct(
    TransformationHelper $transformation_helper,
    ProseMirrorRenderingPluginManager $plugin_manager,
    RendererInterface $renderer,
  ) {
    $this->transformationHelper = $transformation_helper;
    $this->pluginManager = $plugin_manager;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public function render(array $content, string $format = 'html'): string {
    // Validate and enrich the content first.
    $elementInstance = $this->transformationHelper->validateAndSanitize($content);

    if (!$elementInstance->isValid()) {
      // Log errors but continue with original content.
      \Drupal::logger('prosemirror')->warning('Invalid ProseMirror content during rendering: @errors', [
        '@errors' => implode(', ', array_map(fn($e) => $e->getMessage(), $elementInstance->getErrors())),
      ]);
    }

    // Use the sanitized data for rendering.
    $sanitizedContent = $elementInstance->getData();

    // Render the content to a render array.
    $contentRenderArray = $this->renderNode($sanitizedContent, $format);

    // Wrap in ProseMirror-rendered container.
    $renderArray = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => ['class' => ['ProseMirror-rendered']],
      '#attached' => ['library' => ['prosemirror/html-rendering']],
      'content' => $contentRenderArray,
    ];

    // Convert render array to HTML.
    return (string) $this->renderer->renderPlain($renderArray);
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedFormats(): array {
    return ['html', 'plain_text', 'markdown'];
  }

  /**
   * Renders a single node to a render array.
   *
   * @param array $node
   *   The node to render.
   * @param string $format
   *   The target format.
   *
   * @return array
   *   The render array.
   */
  protected function renderNode(array $node, string $format): array {
    if (!isset($node['type'])) {
      return [];
    }

    $type = $node['type'];

    // Try to find a rendering plugin for this node type (use sorted definitions for weight)
    $plugins = $this->pluginManager->getDefinitionsSorted();

    foreach ($plugins as $plugin_id => $definition) {
      if (in_array($type, $definition['node_types'] ?? [])) {
        try {
          $plugin = $this->pluginManager->createInstance($plugin_id);
          $renderArray = [];

          if ($plugin->render($node, $renderArray, $format, $this)) {
            return $renderArray;
          }
        }
        catch (\Exception $e) {
          \Drupal::logger('prosemirror')->error('Error in rendering plugin @plugin: @error', [
            '@plugin' => $plugin_id,
            '@error' => $e->getMessage(),
          ]);
        }
      }
    }

    // Fallback: basic rendering.
    return $this->renderNodeFallback($node, $format);
  }

  /**
   * Fallback rendering for nodes without specific plugins.
   *
   * @param array $node
   *   The node to render.
   * @param string $format
   *   The target format.
   *
   * @return array
   *   The render array.
   */
  protected function renderNodeFallback(array $node, string $format): array {
    $type = $node['type'];

    // Handle other nodes with content.
    $children = [];
    if (isset($node['content']) && is_array($node['content'])) {
      foreach ($node['content'] as $child) {
        $children[] = $this->renderNode($child, $format);
      }
    }

    // Basic wrapper based on node type.
    $tag = $this->getDefaultTagForNodeType($type);

    return [
      '#type' => 'html_tag',
      '#tag' => $tag,
      '#attributes' => ['class' => ['pm-' . str_replace('_', '-', $type)]],
      'children' => $children,
    ];
  }

  /**
   * Gets the default HTML tag for a node type.
   *
   * @param string $type
   *   The node type.
   *
   * @return string
   *   The HTML tag.
   */
  protected function getDefaultTagForNodeType(string $type): string {
    $tagMap = [
      'doc' => 'div',
      'paragraph' => 'p',
      'heading_1' => 'h1',
      'heading_2' => 'h2',
      'heading_3' => 'h3',
      'heading_4' => 'h4',
      'heading_5' => 'h5',
      'heading_6' => 'h6',
      'code_block' => 'pre',
      'bullet_list' => 'ul',
      'ordered_list' => 'ol',
      'list_item' => 'li',
      'blockquote' => 'blockquote',
    ];

    return $tagMap[$type] ?? 'div';
  }

  /**
   * Renders child content for use by plugins.
   *
   * @param array $content
   *   The content array.
   * @param string $format
   *   The target format.
   *
   * @return array
   *   Array of rendered children.
   */
  public function renderChildren(array $content, string $format): array {
    $children = [];
    foreach ($content as $child) {
      $children[] = $this->renderNode($child, $format);
    }
    return $children;
  }

}
