<?php

declare(strict_types=1);

namespace Drupal\prosemirror\Transformation;

/**
 * Represents an intermediate representation of ProseMirror elements.
 *
 * This class holds the intermediate data structure used for:
 * - Serialization and deserialization of content
 * - Transformation between formats
 * - Validation of content structure
 * - Entity reference extraction.
 */
class ElementInstance {

  /**
   * The element data in intermediate format.
   *
   * @var array
   */
  protected array $data;

  /**
   * Entity references found in the element.
   *
   * @var \Drupal\prosemirror\Transformation\EntityReference[]
   */
  protected array $references;

  /**
   * Validation errors encountered.
   *
   * @var \Drupal\prosemirror\Transformation\ValidationError[]
   */
  protected array $errors;

  /**
   * Constructs an ElementInstance.
   *
   * @param array $data
   *   The element data.
   * @param \Drupal\prosemirror\Transformation\EntityReference[] $references
   *   Entity references.
   * @param \Drupal\prosemirror\Transformation\ValidationError[] $errors
   *   Validation errors.
   */
  public function __construct(array $data = [], array $references = [], array $errors = []) {
    $this->data = $data;
    $this->references = $references;
    $this->errors = $errors;
  }

  /**
   * Gets the element data.
   *
   * @return array
   *   The element data.
   */
  public function getData(): array {
    return $this->data;
  }

  /**
   * Gets the entity references.
   *
   * @return \Drupal\prosemirror\Transformation\EntityReference[]
   *   The entity references.
   */
  public function getReferences(): array {
    return $this->references;
  }

  /**
   * Gets the validation errors.
   *
   * @return \Drupal\prosemirror\Transformation\ValidationError[]
   *   The validation errors.
   */
  public function getErrors(): array {
    return $this->errors;
  }

  /**
   * Checks if the element is valid (no errors).
   *
   * @return bool
   *   TRUE if no errors, FALSE otherwise.
   */
  public function isValid(): bool {
    return empty($this->errors);
  }

  /**
   * Merges another element instance into this one.
   *
   * @param \Drupal\prosemirror\Transformation\ElementInstance $other
   *   The other element instance to merge.
   */
  public function merge(ElementInstance $other): void {
    $this->references = array_merge($this->references, $other->getReferences());
    $this->errors = array_merge($this->errors, $other->getErrors());
  }

}
