<?php

declare(strict_types=1);

namespace Drupal\prosemirror\Transformation;

/**
 * Represents a validation error during transformation.
 */
class ValidationError {

  /**
   * The error message.
   *
   * @var string
   */
  protected string $message;

  /**
   * The path to the node where the error occurred.
   *
   * @var array
   */
  protected array $path;

  /**
   * The error severity (error, warning, info).
   *
   * @var string
   */
  protected string $severity;

  /**
   * Additional context data.
   *
   * @var array
   */
  protected array $context;

  /**
   * Constructs a ValidationError.
   *
   * @param string $message
   *   The error message.
   * @param array $path
   *   The path to the node (e.g., ['content', 0, 'content', 1]).
   * @param string $severity
   *   The severity level (default: 'error').
   * @param array $context
   *   Additional context data.
   */
  public function __construct(string $message, array $path = [], string $severity = 'error', array $context = []) {
    $this->message = $message;
    $this->path = $path;
    $this->severity = $severity;
    $this->context = $context;
  }

  /**
   * Gets the error message.
   *
   * @return string
   *   The error message.
   */
  public function getMessage(): string {
    return $this->message;
  }

  /**
   * Gets the path to the error.
   *
   * @return array
   *   The path array.
   */
  public function getPath(): array {
    return $this->path;
  }

  /**
   * Gets the severity.
   *
   * @return string
   *   The severity level.
   */
  public function getSeverity(): string {
    return $this->severity;
  }

  /**
   * Gets the context data.
   *
   * @return array
   *   The context data.
   */
  public function getContext(): array {
    return $this->context;
  }

  /**
   * Converts to array for JSON serialization.
   *
   * @return array
   *   The array representation.
   */
  public function toArray(): array {
    return [
      'message' => $this->message,
      'path' => $this->path,
      'severity' => $this->severity,
      'context' => $this->context,
    ];
  }

  /**
   * Creates a validation error with a formatted path string.
   *
   * @param string $message
   *   The error message.
   * @param array $path
   *   The path array.
   * @param string $severity
   *   The severity level.
   * @param array $context
   *   Additional context.
   *
   * @return static
   *   A new ValidationError instance.
   */
  public static function atPath(string $message, array $path, string $severity = 'error', array $context = []): static {
    $formattedPath = implode('.', array_map(function ($segment) {
      return is_numeric($segment) ? "[{$segment}]" : $segment;
    }, $path));

    $fullMessage = empty($path) ? $message : "{$message} at path: {$formattedPath}";

    return new static($fullMessage, $path, $severity, $context);
  }

}
