<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Plugin\ProseMirror\ElementType;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\prosemirror\Element\ElementProvider;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Plugin\ProseMirror\ElementType\SystemElement;
use Drupal\prosemirror\Transformation\TransformationHelper;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\prosemirror\Transformation\EntityReference;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;
use Prophecy\Argument;

/**
 * Tests the SystemElement plugin.
 *
 * @group prosemirror
 * @coversDefaultClass \Drupal\prosemirror\Plugin\ProseMirror\ElementType\SystemElement
 */
class SystemElementTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The system element plugin.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirror\ElementType\SystemElement
   */
  protected SystemElement $systemElement;

  /**
   * Mock transformation helper.
   *
   * @var \Drupal\prosemirror\Transformation\TransformationHelper|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $transformationHelper;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create service container.
    $container = new ContainerBuilder();
    \Drupal::setContainer($container);

    // Create system element plugin.
    $this->systemElement = new SystemElement([], 'system_element', []);

    // Mock transformation helper.
    $this->transformationHelper = $this->prophesize(TransformationHelper::class);

    // Mock entity repository for media validation.
    $entityRepository = $this->prophesize(EntityRepositoryInterface::class);
    $this->transformationHelper->getEntityRepository()
      ->willReturn($entityRepository->reveal());
  }

  /**
   * Tests doc node validation with valid data.
   *
   * @covers ::validateNode
   */
  public function testValidDocNode(): void {
    $element = $this->prophesize(ProseMirrorElement::class);
    $element->id()->willReturn('doc');

    $node = [
      'type' => 'doc',
      'content' => [
        [
          'type' => 'paragraph',
          'content' => [
            ['type' => 'text', 'text' => 'Hello world'],
          ],
        ],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Mock the transformation helper to handle child nodes.
    $this->transformationHelper->validateChildNode(
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array')
    )->will(function ($args) {
      $child = $args[0];
      // Return the child as-is for simplicity.
      return $child;
    });

    $result = $this->systemElement->validateNode($node, $path, $errors, $references, $this->transformationHelper->reveal());

    $this->assertEquals('doc', $result['type']);
    $this->assertNotEmpty($result['content']);
    $this->assertEmpty($errors);
    $this->assertEmpty($references);
  }

  /**
   * Tests text node validation with valid data.
   *
   * @covers ::validateNode
   */
  public function testValidTextNode(): void {
    $element = $this->prophesize(ProseMirrorElement::class);
    $element->id()->willReturn('text');

    $node = [
      'type' => 'text',
      'text' => 'Hello world',
    ];

    $errors = [];
    $references = [];
    $path = [];

    $result = $this->systemElement->validateNode($node, $path, $errors, $references, $this->transformationHelper->reveal());

    $this->assertEquals('text', $result['type']);
    $this->assertEquals('Hello world', $result['text']);
    $this->assertEmpty($errors);
    $this->assertEmpty($references);
  }

  /**
   * Tests text node validation with marks.
   *
   * @covers ::validateNode
   */
  public function testValidTextNodeWithMarks(): void {
    $element = $this->prophesize(ProseMirrorElement::class);
    $element->id()->willReturn('text');

    $node = [
      'type' => 'text',
      'text' => 'Hello world',
      'marks' => [
        ['type' => 'bold'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Mock the transformation helper to handle marks.
    $this->transformationHelper->sanitizeMarks(
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array')
    )->will(function ($args) {
      $marks = $args[0];
      // Return the marks as-is for simplicity.
      return $marks;
    });

    $result = $this->systemElement->validateNode($node, $path, $errors, $references, $this->transformationHelper->reveal());

    $this->assertEquals('text', $result['type']);
    $this->assertEquals('Hello world', $result['text']);
    $this->assertNotEmpty($result['marks']);
    $this->assertEmpty($errors);
    $this->assertEmpty($references);
  }

  /**
   * Tests text node validation with empty text.
   *
   * @covers ::validateNode
   */
  public function testValidTextNodeWithEmptyText(): void {
    $element = $this->prophesize(ProseMirrorElement::class);
    $element->id()->willReturn('text');

    $node = [
      'type' => 'text',
      'text' => '',
    ];

    $errors = [];
    $references = [];
    $path = [];

    $result = $this->systemElement->validateNode($node, $path, $errors, $references, $this->transformationHelper->reveal());

    $this->assertEquals('text', $result['type']);
    $this->assertEquals('', $result['text']);
    $this->assertEmpty($errors);
    $this->assertEmpty($references);
  }

  /**
   * Tests that non-system elements are rejected.
   *
   * @covers ::validateNode
   */
  public function testNonSystemElementRejected(): void {
    $element = $this->prophesize(ProseMirrorElement::class);
    $element->id()->willReturn('paragraph');

    $node = [
      'type' => 'paragraph',
      'content' => [
        ['type' => 'text', 'text' => 'Hello world'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    $result = $this->systemElement->validateNode($node, $path, $errors, $references, $this->transformationHelper->reveal());

    $this->assertEquals([], $result);
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('System element type plugin received non-system type: paragraph', $errors[0]->getMessage());
  }

  /**
   * Tests that unknown system elements are rejected.
   *
   * @covers ::validateNode
   */
  public function testUnknownSystemElementRejected(): void {
    $element = $this->prophesize(ProseMirrorElement::class);
    $element->id()->willReturn('unknown_system');

    $node = [
      'type' => 'unknown_system',
      'content' => [],
    ];

    $errors = [];
    $references = [];
    $path = [];

    $result = $this->systemElement->validateNode($node, $path, $errors, $references, $this->transformationHelper->reveal());

    $this->assertEquals([], $result);
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('System element type plugin received non-system type: unknown_system', $errors[0]->getMessage());
  }

  /**
   * Tests doc node with invalid content.
   *
   * @covers ::validateNode
   */
  public function testDocNodeWithInvalidContent(): void {
    $element = $this->prophesize(ProseMirrorElement::class);
    $element->id()->willReturn('doc');

    $node = [
      'type' => 'doc',
      'content' => [
        [
          'type' => 'invalid_node',
          'content' => [],
        ],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Mock the transformation helper to return empty for invalid nodes.
    $this->transformationHelper->validateChildNode(
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array')
    )->willReturn([]);

    $result = $this->systemElement->validateNode($node, $path, $errors, $references, $this->transformationHelper->reveal());

    $this->assertEquals('doc', $result['type']);
    // Invalid content should be filtered out.
    $this->assertEmpty($result['content']);
    $this->assertEmpty($errors);
    $this->assertEmpty($references);
  }

}
