<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit;

use Drupal\Core\Entity\EntityInterface;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Entity\ProseMirrorMark;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Prophecy\Argument;

/**
 * Provides test element and mark configurations based on the examples module.
 */
trait ProseMirrorElementTestTrait {

  /**
   * Valid UUIDs for testing entity references.
   */
  protected const VALID_UUIDS = [
    '550e8400-e29b-41d4-a716-446655440000',
    '550e8400-e29b-41d4-a716-446655440001',
    '550e8400-e29b-41d4-a716-446655440002',
    '550e8400-e29b-41d4-a716-446655440003',
    '550e8400-e29b-41d4-a716-446655440004',
  ];

  /**
   * Invalid UUIDs for testing entity references.
   */
  protected const INVALID_UUIDS = [
    'invalid-uuid',
    'not-a-uuid',
    '123',
    'abc-def-ghi',
  ];

  /**
   * Creates mock entity type manager and repository for testing.
   *
   * @return array
   *   Array containing [entityTypeManager, entityRepository] mocks.
   */
  protected function createMockEntityServices() {
    $entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $entityRepository = $this->prophesize(EntityRepositoryInterface::class);

    // Create a mock storage for prosemirror_mark.
    $mockStorage = $this->prophesize(EntityStorageInterface::class);

    // Make the entity type manager return the mock storage for prosemirror_mark.
    $entityTypeManager->getStorage('prosemirror_mark')
      ->willReturn($mockStorage->reveal());

    // Create mock entities for valid UUIDs.
    $mockEntities = [];
    foreach (self::VALID_UUIDS as $uuid) {
      $entity = $this->prophesize(EntityInterface::class);
      $entity->uuid()->willReturn($uuid);
      $mockEntities[$uuid] = $entity->reveal();
    }

    // Mock the entity repository to fake loadEntityByUuid.
    $entityRepository->loadEntityByUuid(Argument::any(), Argument::any())
      ->will(function ($args) use ($mockEntities) {
        $entityType = $args[0];
        $uuid = $args[1];
        // Return a mock entity for valid UUIDs, null for invalid ones.
        return $mockEntities[$uuid] ?? NULL;
      });

    return [$entityTypeManager, $entityRepository];
  }

  /**
   * Creates mock element configurations based on the examples module.
   *
   * @return array
   *   Array of mock ProseMirrorElement entities keyed by ID.
   */
  protected function createMockElements(): array {
    $elements = [];

    // System elements (only doc and text)
    $systemElements = [
      'doc' => ['type' => 'system'],
      'text' => ['type' => 'system'],
    ];

    foreach ($systemElements as $id => $config) {
      $element = $this->prophesize(ProseMirrorElement::class);
      $element->id()->willReturn($id);
      $element->getType()->willReturn($config['type']);
      $element->getOptions()->willReturn($config['options'] ?? []);
      $element->getContentMin()->willReturn($config['content_min'] ?? NULL);
      $element->getContentMax()->willReturn($config['content_max'] ?? NULL);
      $elements[$id] = $element->reveal();
    }

    // Base node elements from examples module.
    $baseNodeElements = [
      'paragraph' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'p', 'marks' => '_'],
      ],
      'blockquote' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'blockquote', 'defining' => TRUE, 'marks' => '_'],
      ],
      'hr' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'hr', 'marks' => ''],
      ],
      'heading_1' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'h1', 'level' => 1, 'defining' => TRUE, 'marks' => '_'],
      ],
      'heading_2' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'h2', 'level' => 2, 'defining' => TRUE, 'marks' => '_'],
      ],
      'heading_3' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'h3', 'level' => 3, 'defining' => TRUE, 'marks' => '_'],
      ],
      'heading_4' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'h4', 'level' => 4, 'defining' => TRUE, 'marks' => '_'],
      ],
      'heading_5' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'h5', 'level' => 5, 'defining' => TRUE, 'marks' => '_'],
      ],
      'heading_6' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'h6', 'level' => 6, 'defining' => TRUE, 'marks' => '_'],
      ],
    ];

    foreach ($baseNodeElements as $id => $config) {
      $element = $this->prophesize(ProseMirrorElement::class);
      $element->id()->willReturn($id);
      $element->getType()->willReturn($config['type']);
      $element->getGroups()->willReturn($config['groups'] ?? []);
      $element->getOptions()->willReturn($config['options'] ?? []);
      $element->getContentMin()->willReturn($config['content_min'] ?? NULL);
      $element->getContentMax()->willReturn($config['content_max'] ?? NULL);
      $elements[$id] = $element->reveal();
    }

    // Custom elements from examples module.
    $customElements = [
      'info_box' => [
        'type' => 'leaf_block',
        'groups' => ['block', 'leaf_block'],
        'options' => [
          'editable' => TRUE,
          'variants' => [
            'warning' => 'Warning',
            'info' => 'Information',
            'success' => 'Success',
            'error' => 'Error',
          ],
        ],
      ],
      'steps' => [
        'type' => 'list_block',
        'groups' => ['block', 'nested'],
        'options' => [
          'direction' => 'column',
          'maxChildren' => 10,
          'contentBlock' => 'step',
        ],
      ],
      'step' => [
        'type' => 'leaf_block',
        'groups' => [],
        'options' => [],
      ],
      'columns' => [
        'type' => 'list_block',
        'groups' => ['block', 'nested'],
        'options' => [
          'direction' => 'row',
          'maxChildren' => 4,
          'contentBlock' => 'column',
        ],
      ],
      'column' => [
        'type' => 'leaf_block',
        'groups' => [],
        'options' => [],
      ],
    ];

    foreach ($customElements as $id => $config) {
      $element = $this->prophesize(ProseMirrorElement::class);
      $element->id()->willReturn($id);
      $element->getType()->willReturn($config['type']);
      $element->getGroups()->willReturn($config['groups'] ?? []);
      $element->getOptions()->willReturn($config['options'] ?? []);
      $element->getContentMin()->willReturn($config['content_min'] ?? NULL);
      $element->getContentMax()->willReturn($config['content_max'] ?? NULL);
      $elements[$id] = $element->reveal();
    }

    // Common configured elements that would normally exist.
    $commonElements = [
      'heading' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'h1', 'defining' => TRUE, 'marks' => '_'],
      ],
      'bullet_list' => [
        'type' => 'base_node',
        'groups' => ['block'],
        'options' => ['tag' => 'ul'],
      ],
      'ordered_list' => [
        'type' => 'base_node',
        'groups' => ['block'],
        'options' => ['tag' => 'ol'],
      ],
      'list_item' => [
        'type' => 'base_node',
        'groups' => [],
        'options' => ['tag' => 'li'],
      ],
      'media' => [
        'type' => 'base_node',
        'groups' => ['inline', 'block'],
        'options' => [],
      ],
      'icon' => [
        'type' => 'base_node',
        'groups' => ['inline'],
        'options' => [],
      ],
      'code_block' => [
        'type' => 'base_node',
        'groups' => ['block'],
        'options' => ['tag' => 'pre'],
      ],
      'table' => [
        'type' => 'base_node',
        'groups' => ['block'],
        'options' => ['tag' => 'table'],
      ],
      'table_row' => [
        'type' => 'base_node',
        'groups' => ['table_content'],
        'options' => ['tag' => 'tr'],
      ],
      'table_cell' => [
        'type' => 'base_node',
        'groups' => ['table_content'],
        'options' => ['tag' => 'td'],
      ],
      'table_header' => [
        'type' => 'base_node',
        'groups' => ['table_content'],
        'options' => ['tag' => 'th'],
      ],
      'horizontal_rule' => [
        'type' => 'base_node',
        'groups' => ['block', 'leaf'],
        'options' => ['tag' => 'hr', 'marks' => ''],
      ],
    ];

    foreach ($commonElements as $id => $config) {
      $element = $this->prophesize(ProseMirrorElement::class);
      $element->id()->willReturn($id);
      $element->getType()->willReturn($config['type']);
      $element->getGroups()->willReturn($config['groups'] ?? []);
      $element->getOptions()->willReturn($config['options'] ?? []);
      $element->getContentMin()->willReturn($config['content_min'] ?? NULL);
      $element->getContentMax()->willReturn($config['content_max'] ?? NULL);
      $elements[$id] = $element->reveal();
    }

    return $elements;
  }

  /**
   * Creates mock mark configurations based on the examples module.
   *
   * @return array
   *   Array of mock ProseMirrorMark entities keyed by ID.
   */
  protected function createMockMarks(): array {
    $marks = [];

    // Common marks from examples module.
    $commonMarks = [
      'bold' => [
        'attributes' => [],
      ],
      'italic' => [
        'attributes' => [],
      ],
      'underline' => [
        'attributes' => [],
      ],
      'strikethrough' => [
        'attributes' => [],
      ],
      'link' => [
        'attributes' => [
          'href' => ['type' => 'string'],
          'title' => ['type' => 'string'],
          'linkType' => ['type' => 'string'],
          'entityType' => ['type' => 'string'],
          'entityUuid' => ['type' => 'string'],
        ],
      ],
      'code' => [
        'attributes' => [],
      ],
      'subscript' => [
        'attributes' => [],
      ],
      'superscript' => [
        'attributes' => [],
      ],
    ];

    foreach ($commonMarks as $id => $config) {
      $mark = $this->prophesize(ProseMirrorMark::class);
      $mark->id()->willReturn($id);
      $mark->getAttributes()->willReturn($config['attributes'] ?? []);
      $marks[$id] = $mark->reveal();
    }

    return $marks;
  }

  /**
   * Sets up element provider mock with elements and marks.
   *
   * @param \Drupal\prosemirror\Element\ElementProvider|\Prophecy\Prophecy\ObjectProphecy $elementProvider
   *   The element provider mock.
   */
  protected function setupElementProviderMock($elementProvider): void {
    $elements = $this->createMockElements();
    $marks = $this->createMockMarks();

    $elementProvider->getAllElements()
      ->willReturn($elements);

    $elementProvider->getAllMarks()
      ->willReturn($marks);

    // Mock individual getters.
    foreach ($elements as $id => $element) {
      $elementProvider->getElement($id)
        ->willReturn($element);
      $elementProvider->hasElement($id)
        ->willReturn(TRUE);
    }

    foreach ($marks as $id => $mark) {
      $elementProvider->getMark($id)
        ->willReturn($mark);
      $elementProvider->hasMark($id)
        ->willReturn(TRUE);
    }
  }

}
