<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Transformation;

use Drupal\prosemirror\Transformation\ElementInstance;
use Drupal\prosemirror\Transformation\EntityReference;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the ElementInstance class.
 *
 * @group prosemirror
 * @coversDefaultClass \Drupal\prosemirror\Transformation\ElementInstance
 */
class ElementInstanceTest extends UnitTestCase {

  /**
   * Tests constructor and basic getters.
   *
   * @covers ::__construct
   * @covers ::getData
   * @covers ::getReferences
   * @covers ::getErrors
   * @covers ::isValid
   */
  public function testConstructorAndGetters(): void {
    // Test with valid data.
    $data = ['type' => 'paragraph', 'content' => [['type' => 'text', 'text' => 'Hello']]];
    $references = [
      new EntityReference('node', '550e8400-e29b-41d4-a716-446655440000'),
    ];
    $errors = [];

    $instance = new ElementInstance($data, $references, $errors);

    $this->assertEquals($data, $instance->getData());
    $this->assertEquals($references, $instance->getReferences());
    $this->assertEquals($errors, $instance->getErrors());
    $this->assertTrue($instance->isValid());

    // Test with errors.
    $errorsWithData = [
      new ValidationError('Test error', ['content', 0]),
    ];

    $instanceWithErrors = new ElementInstance($data, $references, $errorsWithData);
    $this->assertFalse($instanceWithErrors->isValid());
    $this->assertEquals($errorsWithData, $instanceWithErrors->getErrors());
  }

  /**
   * Tests the merge method.
   *
   * @covers ::merge
   */
  public function testMerge(): void {
    // Create first instance.
    $data1 = ['type' => 'paragraph', 'content' => []];
    $references1 = [
      new EntityReference('node', '550e8400-e29b-41d4-a716-446655440001'),
    ];
    $errors1 = [
      new ValidationError('Error 1', ['path', 'to', 'error1']),
    ];
    $instance1 = new ElementInstance($data1, $references1, $errors1);

    // Create second instance.
    $data2 = ['type' => 'heading', 'attrs' => ['level' => 2]];
    $references2 = [
      new EntityReference('media', '550e8400-e29b-41d4-a716-446655440002'),
      new EntityReference('node', '550e8400-e29b-41d4-a716-446655440003'),
    ];
    $errors2 = [
      new ValidationError('Error 2', ['path', 'to', 'error2']),
    ];
    $instance2 = new ElementInstance($data2, $references2, $errors2);

    // Merge instance2 into instance1.
    $instance1->merge($instance2);

    // Check merged references (should contain all references from both)
    $mergedReferences = $instance1->getReferences();
    $this->assertCount(3, $mergedReferences);

    // Check merged errors (should contain all errors)
    $mergedErrors = $instance1->getErrors();
    $this->assertCount(2, $mergedErrors);
    $this->assertEquals('Error 1', $mergedErrors[0]->getMessage());
    $this->assertEquals('Error 2', $mergedErrors[1]->getMessage());

    // Data should still be from instance1 (merge doesn't change data)
    $this->assertEquals($data1, $instance1->getData());

    // Validity should be false if any errors exist.
    $this->assertFalse($instance1->isValid());
  }

  /**
   * Tests merging with duplicate references.
   *
   * @covers ::merge
   */
  public function testMergeWithDuplicateReferences(): void {
    // Create instances with some duplicate references.
    $ref1 = new EntityReference('node', '550e8400-e29b-41d4-a716-446655440001');
    $ref2 = new EntityReference('media', '550e8400-e29b-41d4-a716-446655440002');
    // Duplicate of ref1.
    $ref3 = new EntityReference('node', '550e8400-e29b-41d4-a716-446655440001');

    $instance1 = new ElementInstance([], [$ref1, $ref2], []);
    // Has duplicate.
    $instance2 = new ElementInstance([], [$ref3], []);

    $instance1->merge($instance2);

    // Should have all 3 references (merge doesn't deduplicate)
    $mergedReferences = $instance1->getReferences();
    $this->assertCount(3, $mergedReferences);

    // Verify the unique references.
    $foundTypes = [];
    $foundUuids = [];
    foreach ($mergedReferences as $ref) {
      $foundTypes[] = $ref->getEntityType();
      $foundUuids[] = $ref->getEntityUuid();
    }

    $this->assertContains('node', $foundTypes);
    $this->assertContains('media', $foundTypes);
    $this->assertContains('550e8400-e29b-41d4-a716-446655440001', $foundUuids);
    $this->assertContains('550e8400-e29b-41d4-a716-446655440002', $foundUuids);
  }

  /**
   * Tests empty instance behavior.
   *
   * @covers ::__construct
   * @covers ::isValid
   */
  public function testEmptyInstance(): void {
    $instance = new ElementInstance([], [], []);

    $this->assertEquals([], $instance->getData());
    $this->assertEquals([], $instance->getReferences());
    $this->assertEquals([], $instance->getErrors());
    $this->assertTrue($instance->isValid());
  }

}
