<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Transformation;

use Drupal\Tests\UnitTestCase;
use Drupal\prosemirror\Transformation\TransformationHelper;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\prosemirror\Transformation\EntityReference;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\prosemirror\Element\ElementProvider;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Entity\ProseMirrorMark;
use Psr\Log\LoggerInterface;
use Prophecy\Prophecy\ObjectProphecy;
use Prophecy\Argument;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Tests validation error coverage for TransformationHelper.
 *
 * @coversDefaultClass \Drupal\prosemirror\Transformation\TransformationHelper
 * @group prosemirror
 */
class ValidationErrorCoverageTest extends UnitTestCase {

  use \Drupal\Tests\prosemirror\Unit\ProseMirrorElementTestTrait;

  /**
   * The transformation helper service.
   *
   * @var \Drupal\prosemirror\Transformation\TransformationHelper
   */
  protected TransformationHelper $transformationHelper;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityTypeManager;

  /**
   * Mock entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityRepository;

  /**
   * Mock element type manager.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementTypeManager;

  /**
   * Mock element provider.
   *
   * @var \Drupal\prosemirror\Element\ElementProvider|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementProvider;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    [$this->entityTypeManager, $this->entityRepository] = $this->createMockEntityServices();
    $this->elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $this->logger = $this->prophesize(LoggerInterface::class);
    $this->elementProvider = $this->prophesize(ElementProvider::class);

    // Setup elements and marks using trait.
    $this->setupElementProviderMock($this->elementProvider);

    // Create service container.
    $container = new ContainerBuilder();
    $container->set('prosemirror.element_provider', $this->elementProvider->reveal());
    $container->set('entity.repository', $this->entityRepository->reveal());
    \Drupal::setContainer($container);

    // Create transformation helper with mocked dependencies.
    $this->transformationHelper = new TransformationHelper(
      $this->entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $this->elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $this->logger->reveal()
    );
  }

  /**
   * Tests node missing required type property.
   */
  public function testNodeMissingTypeProperty(): void {
    $node = [
      'content' => [
        ['type' => 'text', 'text' => 'Hello'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($node);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertEquals('Node missing required "type" property', $errors[0]->getMessage());
  }

  /**
   * Tests unknown node type.
   */
  public function testUnknownNodeType(): void {
    $node = [
      'type' => 'unknown_node',
      'content' => [
        ['type' => 'text', 'text' => 'Hello'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($node);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertEquals('Unknown node type: unknown_node', $errors[0]->getMessage());
  }

  /**
   * Tests mark missing required type property.
   */
  public function testMarkMissingTypeProperty(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Hello',
          'marks' => [
            [
              'attrs' => ['style' => 'bold'],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Mark missing required "type" property', $errors[0]->getMessage());
  }

  /**
   * Tests unknown mark type.
   */
  public function testUnknownMarkType(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Hello',
          'marks' => [
            ['type' => 'unknown_mark'],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Unknown mark type: unknown_mark', $errors[0]->getMessage());
  }

  /**
   * Tests multiple validation errors in one document.
   */
  public function testMultipleValidationErrors(): void {
    $document = [
      'type' => 'doc',
      'content' => [
        [
          'type' => 'paragraph',
          'content' => [
            [
              'type' => 'text',
              'text' => 'Hello',
              'marks' => [
                [
    // Missing type.
                  'attrs' => ['style' => 'bold'],
                ],
              ],
            ],
            [
              'type' => 'text',
              'text' => 'World',
              'marks' => [
                ['type' => 'unknown_mark'],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($document);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertGreaterThanOrEqual(2, count($errors));

    // Check for specific errors.
    $errorMessages = array_map(function ($error) {
      return $error->getMessage();
    }, $errors);

    $this->assertContains('Mark missing required "type" property at path: content.[0].content.[0].marks.[0]', $errorMessages);
    $this->assertContains('Unknown mark type: unknown_mark at path: content.[0].content.[1].marks.[0]', $errorMessages);
  }

}
