<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Transformation;

use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the ValidationError class.
 *
 * @group prosemirror
 * @coversDefaultClass \Drupal\prosemirror\Transformation\ValidationError
 */
class ValidationErrorTest extends UnitTestCase {

  /**
   * Tests constructor and getters.
   *
   * @covers ::__construct
   * @covers ::getMessage
   * @covers ::getPath
   * @covers ::getSeverity
   * @covers ::getContext
   */
  public function testConstructorAndGetters(): void {
    $message = 'Test error message';
    $path = ['content', 0, 'attrs'];
    $severity = 'error';
    $context = ['field' => 'value'];

    $error = new ValidationError($message, $path, $severity, $context);

    $this->assertEquals($message, $error->getMessage());
    $this->assertEquals($path, $error->getPath());
    $this->assertEquals($severity, $error->getSeverity());
    $this->assertEquals($context, $error->getContext());
  }

  /**
   * Tests default values.
   *
   * @covers ::__construct
   */
  public function testDefaultValues(): void {
    $message = 'Test error';

    $error = new ValidationError($message);

    $this->assertEquals($message, $error->getMessage());
    $this->assertEquals([], $error->getPath());
    $this->assertEquals('error', $error->getSeverity());
    $this->assertEquals([], $error->getContext());
  }

  /**
   * Tests different severity levels.
   *
   * @covers ::__construct
   * @covers ::getSeverity
   */
  public function testSeverityLevels(): void {
    $errorSeverity = new ValidationError('Error message', [], 'error');
    $warningSeverity = new ValidationError('Warning message', [], 'warning');
    $infoSeverity = new ValidationError('Info message', [], 'info');

    $this->assertEquals('error', $errorSeverity->getSeverity());
    $this->assertEquals('warning', $warningSeverity->getSeverity());
    $this->assertEquals('info', $infoSeverity->getSeverity());
  }

  /**
   * Tests toArray method.
   *
   * @covers ::toArray
   */
  public function testToArray(): void {
    $message = 'Test error';
    $path = ['content', 0, 'marks', 1];
    $severity = 'warning';
    $context = ['expected' => 'foo', 'actual' => 'bar'];

    $error = new ValidationError($message, $path, $severity, $context);

    $expected = [
      'message' => $message,
      'path' => $path,
      'severity' => $severity,
      'context' => $context,
    ];

    $this->assertEquals($expected, $error->toArray());
  }

  /**
   * Tests atPath static method.
   *
   * @covers ::atPath
   */
  public function testAtPath(): void {
    $message = 'Node type not found';
    $path = ['content', 2, 'type'];

    $error = ValidationError::atPath($message, $path);

    $this->assertStringContainsString($message, $error->getMessage());
    $this->assertStringContainsString('content.[2].type', $error->getMessage());
    $this->assertEquals($path, $error->getPath());
    $this->assertEquals('error', $error->getSeverity());
  }

  /**
   * Tests atPath with empty path.
   *
   * @covers ::atPath
   */
  public function testAtPathWithEmptyPath(): void {
    $message = 'Document error';

    $error = ValidationError::atPath($message, []);

    // With empty path, message should not be modified.
    $this->assertEquals($message, $error->getMessage());
    $this->assertEquals([], $error->getPath());
  }

  /**
   * Tests atPath with complex path.
   *
   * @covers ::atPath
   */
  public function testAtPathWithComplexPath(): void {
    $message = 'Invalid attribute';
    $path = ['content', 0, 'content', 1, 'marks', 0, 'attrs', 'href'];

    $error = ValidationError::atPath($message, $path);

    $this->assertStringContainsString('content.[0].content.[1].marks.[0].attrs.href', $error->getMessage());
  }

  /**
   * Tests atPath with custom severity and context.
   *
   * @covers ::atPath
   */
  public function testAtPathWithCustomSeverityAndContext(): void {
    $message = 'Deprecation notice';
    $path = ['attrs', 'oldAttribute'];
    $severity = 'warning';
    $context = ['replacement' => 'newAttribute'];

    $error = ValidationError::atPath($message, $path, $severity, $context);

    $this->assertStringContainsString('attrs.oldAttribute', $error->getMessage());
    $this->assertEquals($severity, $error->getSeverity());
    $this->assertEquals($context, $error->getContext());
  }

  /**
   * Tests path formatting with mixed numeric and string keys.
   *
   * @covers ::atPath
   */
  public function testPathFormattingMixedKeys(): void {
    $paths = [
      [['content'], 'content'],
      [['content', 0], 'content.[0]'],
      [['content', 0, 'attrs'], 'content.[0].attrs'],
      [['marks', 2, 'type'], 'marks.[2].type'],
      [['a', 'b', 'c'], 'a.b.c'],
      [[0, 1, 2], '[0].[1].[2]'],
    ];

    foreach ($paths as [$path, $expectedFormat]) {
      $error = ValidationError::atPath('Test', $path);
      $this->assertStringContainsString($expectedFormat, $error->getMessage());
    }
  }

}
