# ProseMirror Module Architecture

This document provides a comprehensive overview of the ProseMirror module's architecture, explaining how the various systems work together to provide a rich text editing experience in Drupal.

## Overview

The ProseMirror module is a sophisticated rich text editor integration for Drupal that provides:

- **Configurable Elements**: Admin-configurable node types with validation and rendering
- **System Elements**: Built-in elements like media, tables, lists, headings
- **Extensible Plugin System**: Multiple plugin types for different functionality
- **Content Transformation**: Validation, sanitization, and enrichment of content
- **Multi-format Rendering**: HTML output with extensible rendering pipeline
- **Entity Integration**: Deep integration with Drupal's entity system

## Architecture Diagram

```
┌─────────────────────────────────────────────────────────────────────────────────┐
│                         PROSEMIRROR MODULE ARCHITECTURE                         │
└─────────────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────────────┐
│                                FRONTEND LAYER                                   │
├─────────────────────────────────────────────────────────────────────────────────┤
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │   ProseMirror   │    │   Extensions    │    │   JavaScript    │              │
│  │     Editor      │◄───┤    (Plugins)    │◄───┤  Configuration  │              │
│  │                 │    │                 │    │                 │              │
│  └─────────────────┘    └─────────────────┘    └─────────────────┘              │
└─────────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────────┐
│                              DRUPAL INTEGRATION                                 │
├─────────────────────────────────────────────────────────────────────────────────┤
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │  Editor Plugin  │    │  Filter Plugin  │    │   Controllers   │              │
│  │  (Editor API)   │    │ (Filter API)    │    │   (Routes)      │              │
│  └─────────────────┘    └─────────────────┘    └─────────────────┘              │
└─────────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────────┐
│                               PLUGIN SYSTEM                                     │
├─────────────────────────────────────────────────────────────────────────────────┤
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │  Element Type   │    │   Extension     │    │   Rendering     │              │
│  │    Plugins      │    │    Plugins      │    │    Plugins      │              │
│  │                 │    │                 │    │                 │              │
│  │ ┌─────────────┐ │    │ ┌─────────────┐ │    │ ┌─────────────┐ │              │
│  │ │  BaseNode   │ │    │ │   Custom    │ │    │ │   System    │ │              │
│  │ │  LeafBlock  │ │    │ │ Extensions  │ │    │ │  Elements   │ │              │
│  │ │  ListBlock  │ │    │ │             │ │    │ │   Markup    │ │              │
│  │ └─────────────┘ │    │ └─────────────┘ │    │ │ ListBlock   │ │              │
│  └─────────────────┘    └─────────────────┘    │ └─────────────┘ │              │
│                                                └─────────────────┘              │
└─────────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────────┐
│                              ENTITY LAYER                                       │
├─────────────────────────────────────────────────────────────────────────────────┤
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │  ProseMirror    │    │  ProseMirror    │    │  ProseMirror    │              │
│  │    Element      │    │ Element Group   │    │     Mark        │              │
│  │   (Config)      │    │   (Config)      │    │   (Config)      │              │
│  └─────────────────┘    └─────────────────┘    └─────────────────┘              │
└─────────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────────┐
│                            ELEMENT PROVIDER                                     │
├─────────────────────────────────────────────────────────────────────────────────┤
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │   Configurable  │    │     System      │    │    Unified      │              │
│  │    Elements     │◄───┤    Elements     │◄───┤     Access      │              │
│  │                 │    │                 │    │                 │              │
│  └─────────────────┘    └─────────────────┘    └─────────────────┘              │
└─────────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────────┐
│                         TRANSFORMATION PIPELINE                                 │
├─────────────────────────────────────────────────────────────────────────────────┤
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │ Transformation  │    │   Validation    │    │   Element       │              │
│  │    Helper       │◄───┤   & Sanitize    │◄───┤   Instance      │              │
│  │                 │    │                 │    │                 │              │
│  └─────────────────┘    └─────────────────┘    └─────────────────┘              │
│                                                                                 │
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │     Entity      │    │   Validation    │    │   Display       │              │
│  │   Reference     │    │     Errors      │    │  Enrichment     │              │
│  │                 │    │                 │    │                 │              │
│  └─────────────────┘    └─────────────────┘    └─────────────────┘              │
└─────────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────────┐
│                            RENDERING SYSTEM                                     │
├─────────────────────────────────────────────────────────────────────────────────┤
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │  ProseMirror    │    │   Plugin        │    │     Render      │              │
│  │   Renderer      │◄───┤   Matching      │◄───┤     Arrays      │              │
│  │                 │    │                 │    │                 │              │
│  └─────────────────┘    └─────────────────┘    └─────────────────┘              │
│                                                                                 │
│  ┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐              │
│  │   Recursive     │    │    Fallback     │    │      HTML       │              │
│  │   Rendering     │    │   Rendering     │    │     Output      │              │
│  │                 │    │                 │    │                 │              │
│  └─────────────────┘    └─────────────────┘    └─────────────────┘              │
└─────────────────────────────────────────────────────────────────────────────────┘

```

## Core Architecture

### 1. Entity Layer

The module defines three main configuration entities that store editor configuration:

#### ProseMirror Elements (`prosemirror_element`)
- **Purpose**: Define custom node types for the editor
- **Interface**: `ProseMirrorElementInterface`
- **Implementation**: `Entity\ProseMirrorElement`
- **Properties**:
  - `type`: Plugin ID for element type behavior
  - `css_classes`: CSS classes for styling
  - `content`: Content model (what can go inside)
  - `content_min/max`: Content count constraints
  - `groups`: Element groups for schema organization
  - `options`: Type-specific configuration

#### ProseMirror Element Groups (`prosemirror_element_group`)
- **Purpose**: Group elements for schema organization
- **Interface**: `ProseMirrorElementGroupInterface`
- **Implementation**: `Entity\ProseMirrorElementGroup`
- **Usage**: Referenced by elements to define content models

#### ProseMirror Marks (`prosemirror_mark`)
- **Purpose**: Define text formatting marks (bold, italic, links, etc.)
- **Interface**: `ProseMirrorMarkInterface`
- **Implementation**: `Entity\ProseMirrorMark`
- **Properties**:
  - `parse_dom`: Rules for parsing HTML to marks
  - `to_dom`: Rules for rendering marks to HTML
  - `attributes`: Mark attribute definitions
  - `variants`: Mark variations

### 2. Plugin System

The module uses multiple plugin systems for different aspects of functionality:

#### Element Type Plugins
- **Location**: `Plugin/ProseMirror/ElementType/`
- **Interface**: `ProseMirrorElementTypeInterface`
- **Base Class**: `ProseMirrorElementTypeBase`
- **Manager**: `ProseMirrorElementTypeManager`
- **Annotation**: `@ProseMirrorElementType`

**Purpose**: Define behavior for element types including:
- Form building for configuration
- JavaScript configuration generation
- Content validation and sanitization
- Attribute enrichment for display

**Key Implementations**:
- `BaseNode`: Standard HTML elements (p, h1-h6, blockquote, etc.)
- `LeafBlock`: Self-closing elements
- `ListBlock`: List container elements

#### Extension Plugins
- **Location**: `Plugin/ProseMirror/Extension/`
- **Interface**: `ProseMirrorExtensionInterface`
- **Base Class**: `ProseMirrorExtensionBase`
- **Manager**: `ProseMirrorExtensionManager`
- **Annotation**: `@ProseMirrorExtension`

**Purpose**: Provide additional JavaScript libraries and functionality that extend the editor.

#### Rendering Plugins
- **Location**: `Plugin/ProseMirror/Rendering/`
- **Interface**: `ProseMirrorRenderingPluginInterface`
- **Base Class**: `ProseMirrorRenderingPluginBase`
- **Manager**: `ProseMirrorRenderingPluginManager`
- **Annotation**: `@ProseMirrorRenderingPlugin`

**Purpose**: Convert ProseMirror JSON to HTML render arrays.

**Key Implementations**:
- `SystemElementsRenderer`: Handles built-in elements
- `MarkupRenderer`: Handles text formatting marks
- `ListBlockRenderer`: Handles list structures

### 3. Element Provider System

#### ElementProvider Service
- **Location**: `Element/ElementProvider`
- **Purpose**: Unified access to both configurable and system elements
- **Key Methods**:
  - `getAllElements()`: Returns all available elements
  - `getElement($name)`: Get specific element by name
  - `getSystemElements()`: Get only system elements
  - `getConfigurableElements()`: Get only configured elements

#### System Elements
- **Location**: `Element/SystemElementTypes`
- **Purpose**: Define built-in elements that don't require configuration
- **Implementation**: `Element/SystemElement` (virtual entity)

**System Elements Include**:
- `doc`: Root document container
- `text`: Text nodes
- `media`: Media embeds
- `table`, `table_row`, `table_cell`, `table_header`: Table structures
- `bullet_list`, `ordered_list`, `list_item`: List structures
- `code_block`: Code blocks with syntax highlighting
- `heading`: Headings (h1-h6)

### 4. Content Transformation Pipeline

#### TransformationHelper Service
- **Location**: `Transformation/TransformationHelper`
- **Purpose**: Central service for content validation, sanitization, and enrichment

**Key Responsibilities**:
- **Validation**: Ensure content matches schema rules
- **Sanitization**: Remove invalid/dangerous content
- **Entity Reference Extraction**: Find and validate entity references
- **Display Enrichment**: Add display-specific attributes for editor

**Key Classes**:
- `ElementInstance`: Represents processed content with validation results
- `EntityReference`: Represents entity references found in content
- `ValidationError`: Represents validation errors with path information

#### Content Flow
1. **Input**: Raw ProseMirror JSON from editor
2. **Validation**: `TransformationHelper::validateAndSanitize()`
3. **Plugin Processing**: Element type plugins validate their nodes
4. **Reference Extraction**: Entity references collected for dependency tracking
5. **Output**: `ElementInstance` with sanitized data, references, and errors

### 5. Rendering System

#### ProseMirrorRenderer Service
- **Location**: `Rendering/ProseMirrorRenderer`
- **Interface**: `ProseMirrorRendererInterface`
- **Purpose**: Convert validated ProseMirror content to HTML

**Rendering Pipeline**:
1. **Validation**: Content is validated/sanitized first
2. **Plugin Matching**: Find appropriate rendering plugin for each node
3. **Recursive Rendering**: Render nodes and their children
4. **Fallback Rendering**: Default rendering for unhandled nodes
5. **HTML Generation**: Convert render arrays to HTML

### 6. Editor Integration

#### Editor Plugin
- **Location**: `Plugin/Editor/ProseMirror`
- **Purpose**: Integrates with Drupal's Editor API
- **Responsibilities**:
  - Generate JavaScript configuration
  - Provide media integration settings
  - Configure entity selector integration
  - Load extension libraries

#### Filter Plugin
- **Location**: `Plugin/Filter/ProseMirrorFilter`
- **Purpose**: Processes ProseMirror JSON during content rendering
- **Features**:
  - Configurable element/mark filtering per text format
  - JSON validation and error handling
  - Integration with rendering pipeline

### 7. Administrative Interface

#### List Builders
- `ProseMirrorElementListBuilder`: Manage elements
- `ProseMirrorElementGroupListBuilder`: Manage element groups
- `ProseMirrorMarkListBuilder`: Manage marks

#### Form Builders
- `Form/ProseMirrorElementForm`: Element configuration forms
- `Form/ProseMirrorElementGroupForm`: Group configuration forms
- `Form/ProseMirrorMarkForm`: Mark configuration forms
- `Form/ProseMirrorSettingsForm`: Global settings

#### Controllers
- `ProseMirrorAdminController`: Admin redirects
- `ProseMirrorMediaController`: Media metadata API

## Data Flow

### 1. Editor Loading
```
User loads editor
    ↓
Editor plugin generates JS config
    ↓
Element Provider loads all elements
    ↓
Element Type plugins build JS config
    ↓
Extension Manager loads libraries
    ↓
Configuration sent to frontend
```

### 2. Content Saving
```
User saves content
    ↓
ProseMirror JSON submitted
    ↓
Filter processes JSON
    ↓
TransformationHelper validates content
    ↓
Element Type plugins validate nodes
    ↓
Entity references extracted
    ↓
Sanitized content stored
```

### 3. Content Rendering
```
Content requested for display
    ↓
Filter processes stored JSON
    ↓
ProseMirrorRenderer invoked
    ↓
Content validated/enriched
    ↓
Rendering plugins process nodes
    ↓
Render arrays generated
    ↓
HTML output produced
```

## Key Design Patterns

### 1. Plugin Architecture
- Multiple specialized plugin types for different concerns
- Annotation-based discovery
- Weighted execution order
- Extensible by other modules

### 2. Entity-Driven Configuration
- Configuration stored as config entities
- Exportable/importable
- Type-safe with interfaces

### 3. Validation Pipeline
- Multi-stage validation process
- Plugin-specific validation logic
- Error collection with path tracking
- Entity reference extraction

### 4. Hybrid Element System
- Configurable elements for custom needs
- System elements for core functionality
- Unified access through ElementProvider
- Override capabilities

### 5. Transformation-Based Processing
- Immutable transformation pipeline
- Clear separation of concerns
- Error handling and recovery
- Display-time enrichment

## Extension Points

### 1. Element Types
Create new element type plugins to define custom node behaviors:
```php
/**
 * @ProseMirrorElementType(
 *   id = "my_element",
 *   label = @Translation("My Element"),
 *   description = @Translation("Custom element type")
 * )
 */
class MyElement extends ProseMirrorElementTypeBase {
  // Implementation
}
```

### 2. Rendering Plugins
Create rendering plugins for custom output:
```php
/**
 * @ProseMirrorRenderingPlugin(
 *   id = "my_renderer",
 *   label = @Translation("My Renderer"),
 *   node_types = {"my_element"},
 *   weight = 0
 * )
 */
class MyRenderer extends ProseMirrorRenderingPluginBase {
  // Implementation
}
```

### 3. Extensions
Add JavaScript functionality:
```php
/**
 * @ProseMirrorExtension(
 *   id = "my_extension",
 *   label = @Translation("My Extension"),
 *   libraries = {"my_module/my_extension"},
 *   weight = 0
 * )
 */
class MyExtension extends ProseMirrorExtensionBase {
  // Implementation
}
```

## Security Considerations

### 1. Content Validation
- All content is validated against configured schema
- Unknown elements/marks are stripped
- Attribute whitelisting by element type plugins
- Entity reference validation

### 2. XSS Prevention
- HTML output is generated via render arrays
- No direct HTML string concatenation
- Drupal's standard XSS filtering applies
- Attribute sanitization in plugins

### 3. Access Control
- Editor access controlled by text format permissions
- Media access validated for embeds
- Entity reference access checked
- Admin UI protected by permissions

## Performance Considerations

### 1. Caching
- Plugin definitions cached
- Element/mark entities cached
- JavaScript configuration cached
- Render output cached via Drupal's cache system

### 2. Lazy Loading
- Elements loaded on-demand
- Plugin instantiation deferred
- Large configurations split across requests

### 3. Optimization
- Minimal JavaScript payload
- Efficient validation algorithms
- Batch processing for bulk operations

## Testing Strategy

### 1. Unit Tests (automated)
- Plugin functionality
- Validation logic
- Transformation helpers
- Entity operations

### 2. Integration Tests (automated)
- Editor integration
- Filter processing
- Rendering pipeline
- Admin interface

### 3. Browser Tests (manual)
- JavaScript functionality
- User workflows
- Cross-browser compatibility
- Accessibility compliance

## Debugging and Development

### 1. Logging
- Validation errors logged
- Plugin loading failures logged
- Rendering errors logged

### 2. Development Tools
- Debug mode for verbose logging
- Configuration export/import
- Schema validation tools

This architecture provides a solid foundation for rich text editing while maintaining Drupal's principles of configurability, extensibility, and security.
