# ProseMirror Extensibility System

The ProseMirror module provides a comprehensive extensibility system that allows other modules to extend editor functionality without versioning conflicts or complex dependencies.

## Overview

The extensibility system provides two main layers:

1. **Raw ProseMirror API Access**: Direct access to all ProseMirror packages to avoid versioning issues
2. **Plugin Registration System**: A structured way to register nodes, marks, plugins, and menu extensions

## Accessing ProseMirror Modules

All ProseMirror packages are exposed globally via `Drupal.ProseMirrorAPI.modules`:

```javascript
// Access any ProseMirror module
const { Schema, NodeSpec } = Drupal.ProseMirrorAPI.modules['prosemirror-model'];
const { Plugin } = Drupal.ProseMirrorAPI.modules['prosemirror-state'];
const { MenuItem } = Drupal.ProseMirrorAPI.modules['prosemirror-menu'];

// Available modules:
// - prosemirror-commands
// - prosemirror-dropcursor  
// - prosemirror-gapcursor
// - prosemirror-history
// - prosemirror-inputrules
// - prosemirror-keymap
// - prosemirror-menu
// - prosemirror-model
// - prosemirror-schema-basic
// - prosemirror-schema-list
// - prosemirror-search
// - prosemirror-state
// - prosemirror-tables
// - prosemirror-view
```

## Plugin Registration

### Registering Node Types

```javascript
Drupal.ProseMirrorAPI.registerNodePlugin({
  name: 'my_custom_node',
  spec: {
    attrs: { customAttr: { default: 'value' } },
    inline: true,
    group: "inline",
    draggable: true,
    toDOM: (node) => ["span", { class: "my-custom-node" }, node.attrs.customAttr],
    parseDOM: [{
      tag: "span.my-custom-node",
      getAttrs: (dom) => ({ customAttr: dom.textContent })
    }]
  },
  menuItem: (schema, appWrapper) => {
    return new Drupal.ProseMirrorAPI.modules['prosemirror-menu'].MenuItem({
      title: "Insert My Node",
      label: "My Node",
      enable: (state) => canInsert(state, schema.nodes.my_custom_node),
      run: (state, dispatch) => {
        const node = schema.nodes.my_custom_node.create({ customAttr: 'Hello!' });
        dispatch(state.tr.replaceSelectionWith(node));
      }
    });
  },
  plugin: (schema, appWrapper) => {
    return new Drupal.ProseMirrorAPI.modules['prosemirror-state'].Plugin({
      // Custom plugin logic here
    });
  }
});
```

### Registering Mark Types

```javascript
Drupal.ProseMirrorAPI.registerMarkPlugin({
  name: 'my_custom_mark',
  spec: {
    toDOM: () => ["span", { class: "my-custom-mark" }, 0],
    parseDOM: [{ tag: "span.my-custom-mark" }]
  },
  menuItem: (schema, appWrapper) => {
    // Return MenuItem for toggling the mark
  },
  plugin: (schema, appWrapper) => {
    // Optional plugin for mark-specific behavior
  }
});
```

### Registering Menu Extensions

```javascript
Drupal.ProseMirrorAPI.registerMenuExtension({
  section: 'insert', // 'format', 'insert', 'table', or 'other'
  priority: 100, // Lower numbers appear first
  menuItem: (schema, appWrapper) => {
    return new Drupal.ProseMirrorAPI.modules['prosemirror-menu'].MenuItem({
      title: "My Custom Action",
      label: "Custom",
      run: (state, dispatch) => {
        // Custom menu action
      }
    });
  }
});
```

### Registering Plugin Extensions

```javascript
Drupal.ProseMirrorAPI.registerPluginExtension({
  name: 'my_custom_plugin',
  plugin: (schema, appWrapper) => {
    return new Drupal.ProseMirrorAPI.modules['prosemirror-state'].Plugin({
      // Plugin implementation
    });
  }
});
```

## Creating an Extension Module

Here's how to create a complete extension module as a **sibling** to the main prosemirror module:

### 1. Module Structure

```
modules/custom/
├── prosemirror/              # Main ProseMirror module
└── my_prosemirror_extension/ # Your extension module (sibling)
    ├── my_prosemirror_extension.info.yml
    ├── my_prosemirror_extension.libraries.yml  
    ├── my_prosemirror_extension.module
    ├── package.json
    ├── webpack.config.js
    ├── tsconfig.json
    ├── js/
    │   └── src/
    │       └── my-extension.ts
    ├── css/
    │   └── my-extension.css
    └── src/
        └── Plugin/
            └── ProseMirror/
                └── Rendering/
                    └── MyElementRenderer.php
```

### 2. Module Info File (my_prosemirror_extension.info.yml)

```yaml
name: 'My ProseMirror Extension'
type: module
description: 'Adds custom functionality to ProseMirror editor'
core_version_requirement: ^10 || ^11
package: ProseMirror
dependencies:
  - prosemirror:prosemirror
```

### 3. JavaScript Entry Point (js/src/my-extension.ts)

**Important**: Since prosemirror/editor is declared as a dependency, the API should be available immediately. No need for DOM ready events or polling:

```typescript
// Since prosemirror/editor is declared as a dependency, the API should be available
if (!window.Drupal || !window.Drupal.ProseMirrorAPI) {
  throw new Error('ProseMirror API not available. Ensure prosemirror/editor library is loaded before my_extension.');
}

// Register your extension
Drupal.ProseMirrorAPI.registerNodePlugin({
  // ... your plugin definition
});
```

### 4. CSS File (css/my-extension.css)

**Important**: All ProseMirror-related CSS must be scoped with ProseMirror classes:

```css
/* ✅ Correct - scoped to ProseMirror */
.ProseMirror .my-custom-element {
  /* styles */
}

.ProseMirror-rendered .my-custom-element {
  /* rendered output styles */
}

.ProseMirror-dialog .my-dialog-content {
  /* dialog styles */
}

/* ❌ Incorrect - not scoped */
.my-custom-element {
  /* This will affect elements outside ProseMirror */
}
```

### 5. Backend Renderer (src/Plugin/ProseMirror/Rendering/MyElementRenderer.php)

```php
<?php

namespace Drupal\my_prosemirror_extension\Plugin\ProseMirror\Rendering;

use Drupal\prosemirror\Plugin\ProseMirror\Rendering\ProseMirrorRenderingPluginBase;

/**
 * @ProseMirrorRenderingPlugin(
 *   id = "my_element",
 *   label = @Translation("My Element"),
 *   description = @Translation("Renders my custom element"),
 *   elements = {"my_custom_node"}
 * )
 */
class MyElementRenderer extends ProseMirrorRenderingPluginBase {
  
  public function render(array $node, array &$renderArray): bool {
    if ($node['type'] !== 'my_custom_node') {
      return FALSE;
    }
    
    // Build render array
    $renderArray = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#value' => $node['attrs']['customAttr'] ?? '',
      '#attributes' => ['class' => ['my-custom-element']],
    ];
    
    return TRUE;
  }
}
```

### 6. Libraries File (my_prosemirror_extension.libraries.yml)

```yaml
extension:
  version: 1.x
  css:
    theme:
      css/my-extension.css: {}
  js:
    js/dist/my-extension.bundle.js: {}
  dependencies:
    - prosemirror/editor
```

### 7. Module File (my_prosemirror_extension.module)

```php
<?php

use Drupal\Core\Form\FormStateInterface;

/**
 * Implements hook_form_alter().
 */
function my_prosemirror_extension_form_alter(&$form, FormStateInterface $form_state, $form_id) {
  // Attach library when ProseMirror is used
  if (isset($form['#attached']['library'])) {
    foreach ($form['#attached']['library'] as $library) {
      if ($library === 'prosemirror/editor') {
        $form['#attached']['library'][] = 'my_prosemirror_extension/extension';
        break;
      }
    }
  }
}
```

## Example: FontAwesome Icons Module

The `prosemirror_fontawesome_icons` module (a **sibling module** to prosemirror on drupal.org) is available as a complete example of the extensibility system. It demonstrates:

- Accessing FontAwesome libraries independently
- Registering a custom node type (icon)
- Creating custom menu items with dialog interactions
- Implementing custom node views
- Providing backend rendering
- Proper module structure and dependencies
- **No race conditions** - with proper dependency management
- **Scoped CSS** - all styles properly scoped to ProseMirror

## Public API Architecture

```
┌─────────────────────────────────────────────────────────────────────────────────┐
│                           PROSEMIRROR PUBLIC APIS                               │
└─────────────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────────────┐
│                              JAVASCRIPT API                                     │
├─────────────────────────────────────────────────────────────────────────────────┤
│                                                                                 │
│  window.Drupal.ProseMirrorAPI                                                   │
│  ├── modules: {                                                                 │
│  │   ├── 'prosemirror-model': { Schema, NodeSpec, MarkSpec, ... }               │
│  │   ├── 'prosemirror-state': { Plugin, PluginKey, ... }                        │
│  │   ├── 'prosemirror-view': { EditorView, Decoration, ... }                    │
│  │   ├── 'prosemirror-commands': { toggleMark, setBlockType, ... }              │
│  │   ├── 'prosemirror-menu': { MenuItem, MenuBarMenu, ... }                     │
│  │   ├── 'prosemirror-keymap': { keymap, ... }                                  │
│  │   ├── 'prosemirror-history': { history, undo, redo, ... }                    │
│  │   ├── 'prosemirror-inputrules': { InputRule, ... }                           │
│  │   ├── 'prosemirror-schema-basic': { nodes, marks }                           │
│  │   ├── 'prosemirror-schema-list': { addListNodes, ... }                       │
│  │   ├── 'prosemirror-tables': { tableNodes, ... }                              │
│  │   ├── 'prosemirror-gapcursor': { gapCursor }                                 │
│  │   ├── 'prosemirror-dropcursor': { dropCursor }                               │
│  │   └── 'prosemirror-search': { searchKeymap, ... }                            │
│  │   }                                                                          │
│  │                                                                              │
│  ├── registerNodePlugin(config): void                                           │
│  │   ├── name: string                                                           │
│  │   ├── spec: NodeSpec                                                         │
│  │   ├── menuItem?: (schema, appWrapper) => MenuItem                            │
│  │   └── plugin?: (schema, appWrapper) => Plugin                                │
│  │                                                                              │
│  ├── registerMarkPlugin(config): void                                           │
│  │   ├── name: string                                                           │
│  │   ├── spec: MarkSpec                                                         │
│  │   ├── menuItem?: (schema, appWrapper) => MenuItem                            │
│  │   └── plugin?: (schema, appWrapper) => Plugin                                │
│  │                                                                              │
│  ├── registerMenuExtension(config): void                                        │
│  │   ├── section: 'format' | 'insert' | 'table' | 'other'                       │
│  │   ├── priority: number                                                       │
│  │   └── menuItem: (schema, appWrapper) => MenuItem                             │
│  │                                                                              │
│  └── registerPluginExtension(config): void                                      │
│      ├── name: string                                                           │
│      └── plugin: (schema, appWrapper) => Plugin                                 │
│                                                                                 │
└─────────────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────────────┐
│                               PHP PLUGIN API                                    │
├─────────────────────────────────────────────────────────────────────────────────┤
│                                                                                 │
│  Element Type Plugins                                                           │
│  ├── @ProseMirrorElementType annotation                                         │
│  ├── ProseMirrorElementTypeInterface                                            │
│  │   ├── getAvailableOptions(): array                                           │
│  │   ├── validateOptions(array $options): array                                 │
│  │   ├── buildJavaScriptConfiguration(ProseMirrorElementInterface): array       │
│  │   ├── buildOptionsForm(array, FormStateInterface, ProseMirrorElement): array │
│  │   ├── validateNode(array, array, array, array, TransformationHelper): array  │
│  │   ├── validateAttributes(array, array, array, array): array                  │
│  │   └── enrichAttributesForEditorDisplay(array, array): array                  │
│  │                                                                              │
│  Extension Plugins                                                              │
│  ├── @ProseMirrorExtension annotation                                           │
│  ├── ProseMirrorExtensionInterface                                              │
│  │   ├── getLibraries(): array                                                  │
│  │   └── getWeight(): int                                                       │
│  │                                                                              │
│  Rendering Plugins                                                              │
│  ├── @ProseMirrorRenderingPlugin annotation                                     │
│  ├── ProseMirrorRenderingPluginInterface                                        │
│  │   ├── render(array, array, string, ProseMirrorRenderer): bool                │
│  │   ├── getNodeTypes(): array                                                  │
│  │   └── getSupportedFormats(): array                                           │
│  │                                                                              │
│  Services Available for Injection                                               │
│  ├── prosemirror.element_provider                                               │
│  ├── prosemirror.transformation_helper                                          │
│  ├── prosemirror.renderer                                                       │
│  ├── plugin.manager.prosemirror_element_type                                    │
│  ├── plugin.manager.prosemirror_extension                                       │
│  └── plugin.manager.prosemirror_rendering                                       │
│                                                                                 │
└─────────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────────┐
│                            CONFIGURATION API                                    │
├─────────────────────────────────────────────────────────────────────────────────┤
│                                                                                 │
│  Config Entities                                                                │
│  ├── prosemirror_element                                                        │
│  │   ├── id, label, type                                                        │
│  │   ├── css_classes, content, content_min, content_max                         │
│  │   ├── groups, options                                                        │
│  │   └── Methods: getType(), getCssClasses(), getContent(), etc.                │
│  │                                                                              │
│  ├── prosemirror_element_group                                                  │
│  │   ├── id, label                                                              │
│  │   └── Methods: getElements()                                                 │
│  │                                                                              │
│  └── prosemirror_mark                                                           │
│      ├── id, label, parse_dom, to_dom                                           │
│      ├── attributes, variants                                                   │
│      └── Methods: getParseDom(), getToDom(), getAttributes(), etc.              │
│                                                                                 │
│  Global Configuration                                                           │
│  ├── prosemirror.settings                                                       │
│  │   ├── root_group: 'block'                                                    │
│  │   ├── text_group: 'inline'                                                   │
│  │   ├── default_element: 'paragraph'                                           │
│  │   ├── code_block_languages: array                                            │
│  │   ├── table_max_rows: int                                                    │
│  │   ├── table_max_columns: int                                                 │
│  │   └── table_allow_header_rows: bool                                          │
│  │                                                                              │
│  Text Format Integration                                                        │
│  └── Filter settings per text format                                            │
│      ├── system_elements: array (enabled elements)                              │
│      └── marks: array (enabled marks)                                           │
│                                                                                 │
└─────────────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────────────┐
│                              EXTENSION FLOW                                     │
├─────────────────────────────────────────────────────────────────────────────────┤
│                                                                                 │
│  1. Module declares dependency on prosemirror:prosemirror                       │
│  2. Module defines PHP plugins (Element Types, Extensions, Rendering)           │
│  3. Module creates JavaScript files that use Drupal.ProseMirrorAPI              │
│  4. Module declares library dependency on prosemirror/editor                    │
│  5. Module attaches library when ProseMirror is used                            │
│  6. JavaScript executes after ProseMirror API is available                      │
│  7. Extensions register nodes, marks, menu items, and plugins                   │
│  8. PHP plugins handle validation, configuration, and rendering                 │
│                                                                                 │
│  Extension Points Summary:                                                      │
│  ├── Frontend: JavaScript API for editor extensions                             │
│  ├── Backend: PHP plugins for validation and rendering                          │
│  ├── Configuration: Config entities for admin-configurable elements             │
│  ├── Content Processing: Transformation and rendering pipeline                  │
│  └── Integration: Drupal Editor API and Filter API integration                  │
│                                                                                 │
└─────────────────────────────────────────────────────────────────────────────────┘  
```

## Best Practices

1. **CSS Scoping**: Always scope CSS to `.ProseMirror`, `.ProseMirror-*` classes
2. **TypeScript**: Use strict typing where possible
3. **Size Management**: Keep large dependencies (like icon libraries in our example) in separate modules
