import { NodeSpec, Node, Fragment } from 'prosemirror-model';
import { Plugin, TextSelection } from 'prosemirror-state';
import { MenuItem } from 'prosemirror-menu';
import { canInsert } from '../import/menu';
import { Schema } from 'prosemirror-model';
import { EditorView } from 'prosemirror-view';
import { createLeafBlockNode } from './leaf-block';
import { BlockView } from '../core/block-view';
import { ComponentDefinition } from '../wrappers/config';

export type ListBlockDirection = 'row' | 'column';

export interface ListBlockConfig {
  name: string;
  displayName: string;
  maxChildren: number;
  direction: ListBlockDirection;
  contentBlock: string;
  classes?: string[];
  content?: string;
  group?: string;
}

export function createListBlock(config: ListBlockConfig): ComponentDefinition {
  // Custom node view to handle list block rendering and child management
  class ListBlockView extends BlockView {
    private addButton: HTMLElement;
    protected initial: boolean = true;
    
    constructor(node: Node, view: EditorView, getPos: () => number | undefined) {
      super(node, view, getPos);
      
      this.dom.className = `block-wrapper list-block ${config.name} direction-${config.direction}`;
      if (config.classes) {
        this.dom.classList.add(...config.classes); 
      }
    
      // Create content wrapper inside body
      this.contentDOM = document.createElement('div');
      this.contentDOM.className = `list-block-content ${config.name}-content`;
      this.bodyDOM.appendChild(this.contentDOM);

      this.addButton = document.createElement('div');
      this.addButton.className = `list-block-child ${config.name}-child add-child`;
      this.addButton.innerText = '+';
      this.addButton.onclick = () => this.addChild();
      this.bodyDOM.appendChild(this.addButton);

      this.update(node);

      this.initial = false;
    }

    private updateChildIndices() {
      const pos = this.getPos();
      if (pos === undefined) return;

      const { state, dispatch } = this.view;
      const tr = state.tr;
      let hasChanges = false;

      // Update indices for all children
      this.node.forEach((child, offset, index) => {
        if (child.type.isInGroup('leaf_block')) {
          const childPos = pos + offset + 1;
          const currentIndex = child.attrs.index;
          const newIndex = index + 1;
          
          if (currentIndex !== newIndex) {
            if (this.initial) {
              // FIXME: This is a hack to set the index during instantiation.
              (child.attrs as any).index = newIndex;
            }
            else {
              tr.setNodeMarkup(childPos, undefined, {
                ...child.attrs,
                index: newIndex
              });
              hasChanges = true;
            }
          }
        }
      });

      if (hasChanges) {
        dispatch(tr);
      }
    }

    protected updateHeader(): void {
      // Update header content based on node attributes
      this.headerTitleDOM.textContent = config.displayName;
    }

    update(node: Node) {
      if (node.type !== this.node.type) return false;
      this.node = node;

      // Update add button visibility
      if (node.childCount < config.maxChildren) {
        if (!this.bodyDOM.contains(this.addButton)) {
          this.bodyDOM.appendChild(this.addButton);
        }
      } else {
        this.addButton.remove();
      }

      // Update child indices
      this.updateChildIndices();

      // Call parent's update method to handle menu state
      return super.update(node);
    }

    addChild() {
      const pos = this.getPos();
      if (pos === undefined) return;
      
      const { state, dispatch } = this.view;
      const { schema } = state;
      
      // Create a new leaf_block node with proper initialization
      const newChild = createLeafBlockNode(schema, config.contentBlock, {
        index: this.node.childCount + 1
      });
      if (!newChild) return;

      const position = pos + this.node.nodeSize-1;
      
      // Insert the new child at the end of the list block
      const tr = state.tr.insert(position, newChild);
      
      // Set cursor position inside the new child (after the opening tag)
      tr.setSelection(TextSelection.create(tr.doc, position));
      
      dispatch(tr);
    }

    destroy() {
      this.addButton.remove();
      super.destroy();
    }
  }

  const spec: NodeSpec = BlockView.extendNodeSpec({
    isolating: true,
    content: config.content || "leaf_block{1,4}",
    group: config.group || "block",
    draggable: true,
    toDOM: node => {
      return ["div", { 
        class: `list-block ${config.name} direction-${config.direction}`,
      }, ["div", { class: `list-block-content ${config.name}-content` }, 0]];
    },
    parseDOM: [{
      tag: `div.${config.name}`,
      getAttrs: () => ({})
    }]
  });

  const plugin = new Plugin({
    props: {
      nodeViews: {
        [config.name]: (node, view, getPos) => new ListBlockView(node, view, getPos)
      }
    },
    appendTransaction: (transactions, oldState, newState) => {
      // Only proceed if the document has changed
      if (!transactions.some(transaction => transaction.docChanged)) return null;

      const tr = newState.tr;
      let hasChanges = false;

      // Find all list blocks and update their children's indices
      newState.doc.descendants((node, pos) => {
        if (node.type.name === config.name) {
          node.forEach((child, offset, index) => {
            if (child.type.isInGroup('leaf_block')) {
              const childPos = pos + offset + 1;
              const currentIndex = child.attrs.index;
              const newIndex = index + 1;
              
              if (currentIndex !== newIndex) {
                tr.setNodeMarkup(childPos, undefined, {
                  ...child.attrs,
                  index: newIndex
                });
                hasChanges = true;
              }
            }
          });
        }
      });

      return hasChanges ? tr : null;
    }
  });

  const menuItem = (schema: Schema) => new MenuItem({
    title: `Insert ${config.displayName}`,
    label: config.displayName,
    enable(state) { 
      return canInsert(state, schema.nodes[config.name]);
    },
    run(state, dispatch) {
      const node = schema.nodes[config.name].create(
        {},
        Fragment.fromArray([
          createLeafBlockNode(schema, config.contentBlock, { index: 1 }),
          createLeafBlockNode(schema, config.contentBlock, { index: 2 })
        ])
      );
      dispatch(state.tr.replaceSelectionWith(node));
    }
  });

  return { spec, plugin, menuItem, name: config.name };
} 