import { NodeSpec, MarkSpec, Schema } from 'prosemirror-model';
import { Plugin } from 'prosemirror-state';
import { MenuItem } from 'prosemirror-menu';
import { ApplicationWrapper } from '../wrappers/interfaces';

export interface ProseMirrorNodePlugin {
  name: string;
  spec: NodeSpec;
  menuItem?: (schema: Schema, appWrapper: ApplicationWrapper) => MenuItem;
  plugin?: (schema: Schema, appWrapper: ApplicationWrapper) => Plugin;
}

export interface ProseMirrorMarkPlugin {
  name: string;
  spec: MarkSpec;
  menuItem?: (schema: Schema, appWrapper: ApplicationWrapper) => MenuItem;
  plugin?: (schema: Schema, appWrapper: ApplicationWrapper) => Plugin;
}

export interface ProseMirrorMenuExtension {
  section: 'format' | 'insert' | 'table' | 'other';
  priority: number; // Lower numbers appear first
  menuItem: (schema: Schema, appWrapper: ApplicationWrapper) => MenuItem;
}

export interface ProseMirrorPluginExtension {
  name: string;
  plugin: (schema: Schema, appWrapper: ApplicationWrapper) => Plugin;
}

export class ProseMirrorPluginRegistry {
  private nodePlugins: Map<string, ProseMirrorNodePlugin> = new Map();
  private markPlugins: Map<string, ProseMirrorMarkPlugin> = new Map();
  private menuExtensions: ProseMirrorMenuExtension[] = [];
  private pluginExtensions: Map<string, ProseMirrorPluginExtension> = new Map();

  /**
   * Register a new node type plugin
   */
  registerNodePlugin(plugin: ProseMirrorNodePlugin): void {
    this.nodePlugins.set(plugin.name, plugin);
  }

  /**
   * Register a new mark plugin
   */
  registerMarkPlugin(plugin: ProseMirrorMarkPlugin): void {
    this.markPlugins.set(plugin.name, plugin);
  }

  /**
   * Register a menu extension
   */
  registerMenuExtension(extension: ProseMirrorMenuExtension): void {
    this.menuExtensions.push(extension);
    // Sort by priority
    this.menuExtensions.sort((a, b) => a.priority - b.priority);
  }

  /**
   * Register a plugin extension
   */
  registerPluginExtension(extension: ProseMirrorPluginExtension): void {
    this.pluginExtensions.set(extension.name, extension);
  }

  /**
   * Get all registered node plugins
   */
  getNodePlugins(): Map<string, ProseMirrorNodePlugin> {
    return new Map(this.nodePlugins);
  }

  /**
   * Get all registered mark plugins
   */
  getMarkPlugins(): Map<string, ProseMirrorMarkPlugin> {
    return new Map(this.markPlugins);
  }

  /**
   * Get all menu extensions for a specific section
   */
  getMenuExtensions(section?: string): ProseMirrorMenuExtension[] {
    if (section) {
      return this.menuExtensions.filter(ext => ext.section === section);
    }
    return [...this.menuExtensions];
  }

  /**
   * Get all plugin extensions
   */
  getPluginExtensions(): Map<string, ProseMirrorPluginExtension> {
    return new Map(this.pluginExtensions);
  }

  /**
   * Clear all registered extensions (useful for testing)
   */
  clear(): void {
    this.nodePlugins.clear();
    this.markPlugins.clear();
    this.menuExtensions.length = 0;
    this.pluginExtensions.clear();
  }
}

// Global registry instance
export const proseMirrorPluginRegistry = new ProseMirrorPluginRegistry(); 