import {wrapItem, blockTypeItem, Dropdown, joinUpItem, liftItem,
       selectParentNodeItem, undoItem, redoItem, icons, MenuItem, MenuElement, MenuItemSpec} from "prosemirror-menu"
import {EditorState, Command} from "prosemirror-state"
import {Schema, NodeType, MarkType, Fragment} from "prosemirror-model"
import {toggleMark} from "prosemirror-commands"
import {wrapInList} from "prosemirror-schema-list"
import { EntitySelectorPluginSettings } from '../plugins/entity-selector';
import { createLinkDialog } from "../plugins/link-dialog";
import { LinkType, LinkVariantConfig } from "../plugins/link";
import { ApplicationWrapper } from "../wrappers/interfaces";

// Helpers to create specific types of items

export function canInsert(state: EditorState, nodeType: NodeType) {
  let $from = state.selection.$from
  for (let d = $from.depth; d >= 0; d--) {
    let index = $from.index(d)
    if ($from.node(d).canReplaceWith(index, index, nodeType)) return true
  }
  return false
}

function cmdItem(cmd: Command, options: Partial<MenuItemSpec>) {
  let passedOptions: MenuItemSpec = {
    label: options.title as string | undefined,
    run: cmd
  }
  for (let prop in options) (passedOptions as any)[prop] = (options as any)[prop]
  if (!options.enable && !options.select)
    passedOptions[options.enable ? "enable" : "select"] = (state: EditorState) => cmd(state)

  return new MenuItem(passedOptions)
}

function markActive(state: EditorState, type: MarkType) {
  let {from, $from, to, empty} = state.selection
  if (empty) return !!type.isInSet(state.storedMarks || $from.marks())
  else return state.doc.rangeHasMark(from, to, type)
}

function markItem(markType: MarkType, options: Partial<MenuItemSpec>) {
  let passedOptions: Partial<MenuItemSpec> = {
    active(state: EditorState) { return markActive(state, markType) }
  }
  for (let prop in options) (passedOptions as any)[prop] = (options as any)[prop]
  return cmdItem(toggleMark(markType), passedOptions)
}

function linkItem(markType: MarkType, settings: EntitySelectorPluginSettings, variants: LinkVariantConfig, appWrapper: ApplicationWrapper) {
  return new MenuItem({
    title: "Add or remove link",
    icon: icons.link,
    active(state: EditorState) { return markActive(state, markType) }, 
    enable(state: EditorState) { return true; /*!state.selection.empty*/ },
    run(state: EditorState, dispatch, view) {
      if (markActive(state, markType)) {
        // If a link exists, remove it from the selection
        const { from, to } = state.selection;
        const tr = state.tr;

        // Remove any existing link marks in the selection
        state.doc.nodesBetween(from, to, (node, pos) => {
          if (node.marks.length > 0) {
            const linkMark = node.marks.find(mark => mark.type === markType);
            if (linkMark) {
              // Get the full range of the mark
              const start = pos;
              const end = pos + node.nodeSize;
              tr.removeMark(start, end, markType);
            }
          }
        });

        dispatch(tr);
        view.focus();
        return;
      }

      // If no link exists, open the dialog to create one
      const text = state.doc.textBetween(state.selection.from, state.selection.to, " ");
      createLinkDialog({
        view,
        markType,
        entitySelectorSettings: settings,
        isNew: true,
        attrs: {
          text,
          href: "",
          linkType: LinkType.EXTERNAL
        },
        variants,
        appWrapper
      });
    }
  })
}

function wrapListItem(nodeType: NodeType, options: Partial<MenuItemSpec>) {
  return cmdItem(wrapInList(nodeType, (options as any).attrs), options)
}

type MenuItemResult = {
  /// A menu item to toggle the [strong mark](#schema-basic.StrongMark).
  toggleStrong?: MenuItem

  /// A menu item to toggle the [emphasis mark](#schema-basic.EmMark).
  toggleEm?: MenuItem

  /// A menu item to toggle the underline mark.
  toggleUnderline?: MenuItem

  /// A menu item to toggle the strike-through mark.
  toggleStrikeThrough?: MenuItem

  /// A menu item to toggle the small mark.
  toggleSmall?: MenuItem

  /// A menu item to toggle the subscript mark.
  toggleSubscript?: MenuItem

  /// A menu item to toggle the superscript mark.
  toggleSuperscript?: MenuItem

  /// A menu item to toggle the [code font mark](#schema-basic.CodeMark).
  toggleCode?: MenuItem

  /// A menu item to toggle the [link mark](#schema-basic.LinkMark).
  toggleLink?: MenuItem

  /// A menu item to insert an [image](#schema-basic.Image).
  insertImage?: MenuItem

  /// A menu item to wrap the selection in a [bullet list](#schema-list.BulletList).
  wrapBulletList?: MenuItem

  /// A menu item to wrap the selection in an [ordered list](#schema-list.OrderedList).
  wrapOrderedList?: MenuItem

  /// A menu item to wrap the selection in a [block quote](#schema-basic.BlockQuote).
  wrapBlockQuote?: MenuItem

  /// A menu item to set the current textblock to be a normal
  /// [paragraph](#schema-basic.Paragraph).
  makeParagraph?: MenuItem

  /// A menu item to set the current textblock to be a
  /// [code block](#schema-basic.CodeBlock).
  makeCodeBlock?: MenuItem

  /// Menu items to set the current textblock to be a
  /// [heading](#schema-basic.Heading) of level _N_.
  makeHead1?: MenuItem
  makeHead2?: MenuItem
  makeHead3?: MenuItem
  makeHead4?: MenuItem
  makeHead5?: MenuItem
  makeHead6?: MenuItem

  /// A menu item to insert a horizontal rule.
  insertHorizontalRule?: MenuItem

  /// A menu item to insert a table.
  insertTable?: MenuItem

  /// A dropdown containing the `insertImage` and
  /// `insertHorizontalRule` items.
  insertMenu: Dropdown

  /// A dropdown containing the items for making the current
  /// textblock a paragraph, code block, or heading.
  typeMenu: Dropdown

  /// Array of block-related menu items.
  blockMenu: MenuElement[][]

  /// Inline-markup related menu items for most-used formatting: bold, italic.
  formatMenu: MenuElement[][]

  /// Inline-markup related drop down for less used formatting.
  advancedFormatMenu: Dropdown

  /// Other inline-markup related menu items.
  inlineMenu: MenuElement[][]

  /// An array of arrays of menu elements for use as the full menu
  /// for, for example the [menu
  /// bar](https://github.com/prosemirror/prosemirror-menu#user-content-menubar).
  fullMenu: MenuElement[][]
}

/// Given a schema, look for default mark and node types in it and
/// return an object with relevant menu items relating to those marks.
export function buildMenuItems(
  schema: Schema, 
  inserts: MenuItem[], 
  settings: {entitySelector: EntitySelectorPluginSettings, rootGroup?: string, systemNodes?: any}, 
  linkVariants: LinkVariantConfig,
  appWrapper: ApplicationWrapper
): MenuItemResult {
  let r: MenuItemResult = {} as any
  let mark: MarkType | undefined
  let node: NodeType | undefined

  let textTypes: {nodeType: NodeType, menuItem: MenuItem}[] = [];

  // Text types - build dynamically based on available nodes
  const textNodeTypes = ['paragraph', 'code_block'];
  
  // Add paragraph
  if (node = schema.nodes.paragraph) {
    const menuItem = blockTypeItem(node, {
      title: "Change to regular, formatted text",
      label: "Normal text"
    });
    r.makeParagraph = menuItem;
    textTypes.push({ nodeType: node, menuItem });
  }
  
  // Add code block
  if (node = schema.nodes.code_block) {
    const menuItem = blockTypeItem(node, {
      title: "Change to code block",
      label: "Code"
    });
    r.makeCodeBlock = menuItem;
    textTypes.push({ nodeType: node, menuItem });
  }
  
  // Handle heading elements from configured components
  const headingItems: MenuItem[] = [];
  
  // Look for configured heading elements in the schema
  Object.keys(schema.nodes).forEach(nodeName => {
    const node = schema.nodes[nodeName];
    const nodeSpec = node.spec;
    
    // Check if this is a heading element (has level attribute and h tag)
    if (nodeSpec.attrs?.level && nodeSpec.toDOM) {
      const level = nodeSpec.attrs.level.default;
      if (level >= 1 && level <= 6) {
        const menuItem = blockTypeItem(node, {
          title: `Change to heading ${level}`,
          label: `Heading ${level}`
        });
        (r as any)[`makeHead${level}`] = menuItem;
        textTypes.push({
          nodeType: node,
          menuItem
        });
        headingItems.push(menuItem);
      }
    }
  });

  // Group text types
  let cut = <T>(arr: T[]) => arr.filter(x => x) as NonNullable<T>[]
  const typeOptions = cut([r.makeParagraph, r.makeCodeBlock, ...headingItems]);
  
  // Create a dynamic label function that returns a string based on current state
  const typeMenuLabel = (state: EditorState): string => {
    const { $from } = state.selection;
    const node = $from.parent;
    
    // Check for specific node types
    if (node.type === schema.nodes.paragraph) {
      return "Normal text";
    } else if (node.type === schema.nodes.code_block) {
      return "Code";
    } else {
      // Check for heading types by looking at level attribute
      if (node.type.spec.attrs?.level && node.attrs.level) {
        const level = node.attrs.level;
        if (level >= 1 && level <= 6) {
          return `Heading ${level}`;
        }
      }
      return "Text";
    }
  };
  
  r.typeMenu = new Dropdown(typeOptions, {label: typeMenuLabel})

  if (mark = schema.marks.bold)
    r.toggleStrong = markItem(mark, {title: "Toggle strong style", icon: icons.strong})
  if (mark = schema.marks.italic)
    r.toggleEm = markItem(mark, {title: "Toggle emphasis", icon: icons.em})
  if(mark = schema.marks.underline)
    r.toggleUnderline = markItem(mark, {title: "Toggle underline", label: "Underline"/*icon: {text: "U", css: "text-decoration: underline"}*/})
  if(mark = schema.marks.strike)
    r.toggleStrikeThrough = markItem(mark, {title: "Toggle strike-through", label: "Strikethrough"/*, icon: {text: "S", css: "text-decoration: line-through"}*/})
  if(mark = schema.marks.small)
    r.toggleSmall = markItem(mark, {title: "Toggle small", label: "Small"/*, icon: {text: "s", css: "font-size: 90%"}*/})
  if(mark = schema.marks.subscript)
    r.toggleSubscript = markItem(mark, {title: "Toggle subscript", label: "Subscript"/*, icon: {text: "s", css: "vertical-align: sub; font-size: 70%"}*/})
  if(mark = schema.marks.superscript)
    r.toggleSuperscript = markItem(mark, {title: "Toggle superscript", label: "Superscript"/*, icon: {text: "s", css: "vertical-align: super; font-size: 70%"}*/})
  if (mark = schema.marks.code)
    r.toggleCode = markItem(mark, {title: "Toggle code font", label: "Code"/*, icon: icons.code*/})
  if (mark = schema.marks.link)
    r.toggleLink = linkItem(mark, settings.entitySelector, linkVariants, appWrapper)

  if (settings.systemNodes?.bullet_list && (node = schema.nodes.bullet_list))
    r.wrapBulletList = wrapListItem(node, {
      title: "Wrap in bullet list",
      icon: icons.bulletList
    })
  if (settings.systemNodes?.ordered_list && (node = schema.nodes.ordered_list))
    r.wrapOrderedList = wrapListItem(node, {
      title: "Wrap in ordered list",
      icon: icons.orderedList
    })
  if (node = schema.nodes.blockquote)
    r.wrapBlockQuote = wrapItem(node, {
      title: "Wrap in block quote",
      icon: icons.blockquote
    })
  if (node = schema.nodes.horizontal_rule) {
    let hr = node
    r.insertHorizontalRule = new MenuItem({
      title: "Insert horizontal rule",
      label: "Horizontal rule",
      icon: {text: /*"▬"*/"—"},
      enable(state: EditorState) { return canInsert(state, hr) },
      run(state: EditorState, dispatch) { dispatch(state.tr.replaceSelectionWith(hr.create())) }
    })
  }
  // Only add table menu if tables are enabled in system configuration
  if (settings.systemNodes?.table && schema.nodes.table && schema.nodes.table_row && schema.nodes.table_cell) {
    const table = schema.nodes.table!
    const header = schema.nodes.table_header
    const row = schema.nodes.table_row!
    const cell = schema.nodes.table_cell!
    
    // Get table settings from backend configuration
    const tableSettings = settings.systemNodes.table;
    const initialColumns = Math.min(tableSettings.max_columns || 10, 3); // Start with 3 columns, max from settings
    const initialRows = Math.min(tableSettings.max_rows || 50, 3); // Start with 3 rows, max from settings
    const allowHeaderRows = tableSettings.allow_header_rows ?? true;
    
    r.insertTable = new MenuItem({
      title: "Insert table",
      label: "Table",
      icon: {text: "▦"},
      enable(state: EditorState) { return canInsert(state, table) },
      run(state: EditorState, dispatch) {
        const cellNode = cell.createAndFill()!;
        const headerCellNode = header?.createAndFill() ?? cellNode;
        
        // Create rows based on settings
        const rows: any[] = [];
        
        for (let rowIndex = 0; rowIndex < initialRows; rowIndex++) {
          const cells: any[] = [];
          
          // Create cells for this row
          for (let colIndex = 0; colIndex < initialColumns; colIndex++) {
            // Use header cells for first row if header rows are allowed
            const useHeader = allowHeaderRows && rowIndex === 0 && header;
            cells.push(useHeader ? headerCellNode : cellNode);
          }
          
          rows.push(row.create(null, Fragment.fromArray(cells)));
        }
        
        const tableNode = table.create(null, Fragment.fromArray(rows))!;
        dispatch(state.tr.replaceSelectionWith(tableNode));
      }
    })
  }

  // Filter inserts based on the root group
  const filteredInserts = inserts.filter(insert => {
    // For now, include all inserts. In the future, we can filter based on whether
    // the node belongs to the root group
    return true;
  });
  
  r.insertMenu = new Dropdown(cut([/*r.insertImage, */...filteredInserts]), {label: "Embed", class: "menu-embed"})

  // Table: ▦, Row: ▥, Column: ▤, Container: ▢ or ▭ or ▯

  r.advancedFormatMenu = new Dropdown(cut([r.toggleUnderline, r.toggleSuperscript, r.toggleSubscript, r.toggleSmall, r.toggleStrikeThrough, r.toggleCode]), {label: "⋯", class: "drop-down-icon"});

  // Build menu sections as requested:
  // 1. Text types
  const textSection = [[r.typeMenu]];
  
  // 2. Undo/redo
  const historySection = [[undoItem, redoItem]];
  
  // 3. Marks (formatting)
  r.formatMenu = [cut([r.toggleStrong, r.toggleEm, r.advancedFormatMenu])];
  
  // 4. Links
  r.inlineMenu = [cut([r.toggleLink])];
  
  // 5. Nested base nodes (blockquotes, lists, tables, hr)
  const nestedNodesSection = [cut([r.wrapBulletList, r.wrapOrderedList, r.wrapBlockQuote, r.insertHorizontalRule, r.insertTable])];
  
  // 6. Structure operations (indentation, joining, selecting parent)
  //const structureSection = [cut([joinUpItem, liftItem, selectParentNodeItem])];
  
  // 7. Embed dropdown
  const embedSection = [[r.insertMenu]];
  
  r.blockMenu = [...nestedNodesSection, /*...structureSection,*/ ...embedSection];
  r.fullMenu = [...textSection, ...historySection, ...r.formatMenu, ...r.inlineMenu, ...r.blockMenu];

  return r
}
