import { Field, FieldOptions } from "../import/prompt";
import { LinkType } from "./link";
import { ApplicationWrapper, AutocompleteItem, AutocompleteFieldOptions } from '../wrappers/interfaces';
import { AutocompleteItem as RendererAutocompleteItem } from '../renderers/interfaces';

export type EntitySelectorProfileType = 'external' | string;

export interface EntitySelectorProfile {
  profile: string;
  autocompleteUrl: string;
  label: string;
}

export interface EntitySelectorPluginSettings {
  profiles: Record<string, EntitySelectorProfile>;
}

export interface EntitySelectorSettings extends FieldOptions {
  type: EntitySelectorProfileType;
  profile: EntitySelectorProfile;
  value: string;
  url?: string;
  entityLabel?: string;
}

/// A field class for single-line text fields.
export class EntitySelectorField extends Field<EntitySelectorSettings> {
  private selectedItem: HTMLElement | null = null;
  private input: HTMLInputElement | null = null;
  private container: HTMLElement | null = null;
  protected element: HTMLInputElement | null = null;

  constructor(
    options: EntitySelectorSettings,
    protected readonly app: ApplicationWrapper,
  ) {
    super(options);
  }

  async querySuggestions(term: string): Promise<AutocompleteItem[]> {
    return new Promise((resolve, reject) => {
      this.app.ajax({
        url: this.options.profile.autocompleteUrl,
        data: { q: term },
        dataType: 'json',
        success: (data: { suggestions: AutocompleteItem[] }) => {
          //console.debug(data);
          resolve(data.suggestions);
        },
        error: (err: any) => {
          console.error(err);
          reject(err);
        },
      });
    });
  }

  render() {
    const renderer = this.app.getRendererFactory();
    
    const containerResult = renderer.elements.div({ className: 'entity-selector-field' });
    const container = this.container = containerResult.element;

    const inputWrapperResult = renderer.elements.div({ className: 'entity-selector-field__input-wrapper' });
    const inputWrapper = inputWrapperResult.element;
    
    const inputLabelResult = renderer.elements.label('Target', {
      attributes: { for: 'entity-selector-url-input' }
    });
    const inputLabel = inputLabelResult.element;

    // Create selected item display
    const selectedItemResult = renderer.elements.div({
      className: 'entity-selector-field__selected-item hidden'
    });
    const selectedItem = this.selectedItem = selectedItemResult.element;

    const selectedLink = document.createElement('a');
    selectedLink.setAttribute('href', '#');
    selectedLink.setAttribute('target', '_blank');
    selectedLink.className = 'entity-selector-field__selected-link';

    const removeButtonResult = renderer.elements.button('Remove selection', {
      className: 'button button--small button--danger entity-selector-field__remove-button',
      listeners: {
        click: (e: Event) => {
          e.preventDefault();
          this.clearSelection();

          // Show autocomplete input again
          if (!this.input) {
            this.createAutocompleteInput(inputWrapper);
          } else {
            this.input.classList.remove('hidden');
          }
          selectedItem.classList.add('hidden');
        }
      }
    });
    const removeButton = removeButtonResult.element;

    selectedItem.appendChild(selectedLink);
    selectedItem.appendChild(removeButton);
    inputWrapper.appendChild(inputLabel);
    container.appendChild(inputWrapper);
    container.appendChild(selectedItem);

    // If we have a value, show it as selected and hide the input
    if (this.options.value) {
      // Set up a dummy element for validation
      const inputResult = renderer.elements.input({ 
        type: 'hidden',
        value: this.options.value 
      });
      const input = inputResult.element;
      if (this.options.value.startsWith('entity:')) {
        const [type, uuid] = this.options.value.replace('entity:', '').split('/');
        input.dataset['linkType'] = LinkType.INTERNAL;
        input.dataset['entityUuid'] = uuid;
        input.dataset['entityType'] = type;
        if (this.options.url) {
          input.dataset['entityUrl'] = this.options.url;
        }
        if (this.options.entityLabel) {
          input.dataset['entityLabel'] = this.options.entityLabel;
        }
      }
      this.element = input;
      inputWrapper.appendChild(input);
      this.showSelectedItem({
        label: this.options.entityLabel || this.options.value,
        url: this.options.url || this.options.value
      });
      selectedItem.classList.remove('hidden');
      // Do not render autocomplete input
      this.input = null;
    } else {
      this.createAutocompleteInput(inputWrapper);
    }

    return container;
  }

  private createAutocompleteInput(inputWrapper: HTMLElement) {
    const renderer = this.app.getRendererFactory();
    const autocompleteResult = renderer.components.autocomplete({
      placeholder: this.options.entityLabel ?? '',
      value: '',
      required: this.options.required,
      onQuery: async (term: string) => {
        const items = await this.querySuggestions(term);
        // Map to renderer's AutocompleteItem format
        return items.map(item => ({
          value: item.path,
          label: item.label,
          description: item.description,
          group: item.group,
          metadata: {
            status: item.status,
            entity_type_id: item.entity_type_id,
            entity_uuid: item.entity_uuid,
            path: item.path,
            substitution_id: item.substitution_id
          }
        }));
      },
      onSelect: (item: RendererAutocompleteItem) => {
        // Extract metadata
        const metadata = item.metadata as any;
        const entity_type_id = metadata?.entity_type_id || '';
        const entity_uuid = metadata?.entity_uuid || '';
        const path = metadata?.path || item.value;
        
        // Format the value as entity:type/uuid
        const value = `entity:${entity_type_id}/${entity_uuid}`;
        this.input!.value = value;
        this.input!.dataset['linkType'] = LinkType.INTERNAL;
        this.input!.dataset['entityUuid'] = entity_uuid;
        this.input!.dataset['entityType'] = entity_type_id;
        this.input!.dataset['entityUrl'] = path;
        this.input!.dataset['entityLabel'] = item.label;
        this.showSelectedItem({ label: item.label, url: path });
        if (this.input) this.input.classList.add('hidden');
        if (this.selectedItem) this.selectedItem.classList.remove('hidden');
        this.container?.dispatchEvent(new CustomEvent('entity-selector:change', { detail: true }));
        this.onSubmit?.();
      }
    });
    
    inputWrapper.appendChild(autocompleteResult.element);
    const input = this.input = autocompleteResult.element.querySelector('input') as HTMLInputElement;
    this.element = input;
  }

  private showSelectedItem(item: { label: string; url: string }) {
    if (this.selectedItem) {
      this.selectedItem.classList.remove('hidden');
      const link = this.selectedItem.querySelector('.entity-selector-field__selected-link') as HTMLAnchorElement;
      if (link) {
        link.textContent = item.label;
        link.href = this.sanitizeUrl(item.url);
      }
    }
    if (this.input) {
      this.input.classList.add('hidden');
    }
  }

  private sanitizeUrl(url: string): string {
    // Prevent javascript: and data: URLs which could be used for XSS
    const dangerousProtocols = ['javascript:', 'data:', 'vbscript:', 'file:', 'about:'];
    const lowerUrl = url.toLowerCase().trim();
    
    for (const protocol of dangerousProtocols) {
      if (lowerUrl.startsWith(protocol)) {
        return '#'; // Return safe fallback
      }
    }
    
    return url;
  }

  private clearSelection() {
    if (this.input && this.selectedItem) {
      this.input.value = '';
      this.input.classList.remove('hidden');
      this.selectedItem.classList.add('hidden');
      delete this.input.dataset['linkType'];
      delete this.input.dataset['entityUuid'];
      delete this.input.dataset['entityType'];
      delete this.input.dataset['entityUrl'];
      delete this.input.dataset['entityLabel'];
      this.container?.dispatchEvent(new CustomEvent('entity-selector:change', { detail: false }));
      this.onSubmit?.();
    }
  }

  read(dom?: HTMLElement) {
    const input = dom ? (dom.querySelector('input') as HTMLInputElement) : this.element;
    if (input) {
      const value = input.value;
      if (value.startsWith('entity:')) {
        const hasAllAttributes = [
          'linkType',
          'entityUuid',
          'entityType',
          'entityUrl',
          'entityLabel'
        ].every(attr => input.dataset[attr] !== undefined);
        if (!hasAllAttributes) {
          return '';
        }
      }
      return value;
    }
    return '';
  }

  isValid(): boolean {
    if (!this.element) return false;
    const value = this.element.value;
    if (!value) return false;
    if (value.startsWith('entity:')) {
      const isValid = [
        'linkType',
        'entityUuid',
        'entityType',
        'entityUrl',
        'entityLabel'
      ].every(attr => this.element!.dataset[attr] !== undefined);
      return isValid;
    }
    return false;
  }
}
