import { EntitySelectorField, EntitySelectorPluginSettings, EntitySelectorProfileType } from './entity-selector';
import { EditorView } from "prosemirror-view"
import { MarkType } from "prosemirror-model"
import { Field, TextField, SelectField, FieldOptions } from "../import/prompt"
import { LinkType, LinkVariantConfig } from "./link"
import { ApplicationWrapper } from '../wrappers/interfaces';

export interface LinkDialogOptions {
  view: EditorView
  markType: MarkType
  entitySelectorSettings: EntitySelectorPluginSettings
  isNew: boolean
  from?: number
  to?: number
  variants: LinkVariantConfig
  attrs?: {
    href: string
    text: string
    linkType: LinkType
    entityUuid?: string
    entityType?: string
    entityUrl?: string
    entityLabel?: string
    linkUri?: string
    variant?: string
  };
  appWrapper: ApplicationWrapper;
}

export function createLinkDialog(options: LinkDialogOptions) {
  const { view, markType, entitySelectorSettings, isNew, attrs, from: providedFrom, to: providedTo, variants, appWrapper } = options;
  const { state } = view;
  let { from, to } = state.selection;

  // Use provided range if available, otherwise use current selection
  if (providedFrom !== undefined && providedTo !== undefined) {
    from = providedFrom;
    to = providedTo;
  }

  const text = attrs?.text || state.doc.textBetween(from, to, " ");
  const href = attrs?.href || "";
  const linkType = attrs?.linkType || (href.startsWith('entity:') ? LinkType.INTERNAL : LinkType.EXTERNAL);
  const variant = attrs?.variant || 'link';

  // Create form and containers using renderer
  const renderer = appWrapper.getRendererFactory();
  const formResult = renderer.elements.form({ className: 'link-dialog-form' });
  const form = formResult.element;
  const fieldsResult = renderer.elements.div({ className: 'link-dialog-fields' });
  const fields = fieldsResult.element;
  const buttonsResult = renderer.elements.div({ className: 'link-dialog-buttons' });
  const buttons = buttonsResult.element;

  // Text field
  const textField = new TextField({
    label: 'Text',
    value: text,
    required: true,
    app: appWrapper
  });
  const textFieldContainerResult = renderer.elements.div({ className: 'form-item' });
  const textFieldContainer = textFieldContainerResult.element;
  const textLabelResult = renderer.elements.label('Text', { className: 'form-label' });
  const textLabel = textLabelResult.element;
  const textInput = textField.render();
  textInput.classList.add('form-text');
  textFieldContainer.appendChild(textLabel);
  textFieldContainer.appendChild(textInput);
  fields.appendChild(textFieldContainer);

  // Type selector
  const typeOptions = [
    { value: 'external', label: 'URL' },
    ...Object.entries(entitySelectorSettings.profiles).map(([key, profile]) => ({
      value: key,
      label: profile.label
    }))
  ];
  
  // Determine default type based on href
  let defaultType: EntitySelectorProfileType = 'external';
  if (linkType === LinkType.INTERNAL && href.startsWith('entity:')) {
    const entityType = href.replace('entity:', '').split('/')[0];
    if (entitySelectorSettings.profiles[entityType]) {
      defaultType = entityType;
    }
  }
  
  const typeField = new SelectField({
    label: 'Type',
    options: typeOptions,
    value: defaultType,
    required: true,
    app: appWrapper
  } as FieldOptions);
  const typeFieldContainerResult = renderer.elements.div({ className: 'form-item' });
  const typeFieldContainer = typeFieldContainerResult.element;
  const typeLabelResult = renderer.elements.label('Type', { className: 'form-label' });
  const typeLabel = typeLabelResult.element;
  const typeSelect = typeField.render();
  typeSelect.classList.add('form-select');
  typeFieldContainer.appendChild(typeLabel);
  typeFieldContainer.appendChild(typeSelect);
  fields.appendChild(typeFieldContainer);

  // Variant selector
  const variantOptions = Object.entries(variants).map(([value, label]) => ({ value, label }));
  let variantSelect: HTMLSelectElement | undefined;
  if (variantOptions.length > 1) {
    const variantField = new SelectField({
      label: 'Style',
      options: variantOptions,
      value: variant,
      required: true,
      app: appWrapper
    } as FieldOptions);
    const variantFieldContainerResult = renderer.elements.div({ className: 'form-item' });
    const variantFieldContainer = variantFieldContainerResult.element;
    const variantLabelResult = renderer.elements.label('Style', { className: 'form-label' });
    const variantLabel = variantLabelResult.element;
    variantSelect = variantField.render() as HTMLSelectElement;
    variantSelect.classList.add('form-select');
    variantFieldContainer.appendChild(variantLabel);
    variantFieldContainer.appendChild(variantSelect);
    fields.appendChild(variantFieldContainer);
  }

  // Target field container
  const targetContainerResult = renderer.elements.div({ className: 'form-item link-dialog-target' });
  const targetContainer = targetContainerResult.element;
  fields.appendChild(targetContainer);

  // Store reference to current EntitySelectorField instance
  let currentEntitySelectorField: EntitySelectorField | null = null;

  // Update target field based on type
  function updateTargetField(type: EntitySelectorProfileType) {
    targetContainer.innerText = '';
    let targetField: Field<any>;

    if (type === 'external') {
      // For external links, use href if it's not an entity: URL, otherwise use empty string
      const externalValue = href.startsWith('entity:') ? '' : href;
      targetField = new TextField({
        label: 'URL',
        value: externalValue,
        required: true,
        app: appWrapper
      });
      const targetLabelResult = renderer.elements.label('URL', { className: 'form-label' });
      const targetLabel = targetLabelResult.element;
      const targetInput = targetField.render();
      targetInput.classList.add('form-text');
      targetContainer.appendChild(targetLabel);
      targetContainer.appendChild(targetInput);
      currentEntitySelectorField = null;
    } else {
      const profile = entitySelectorSettings.profiles[type];
      // Only use href for entity selector if it's already in entity: format and matches the selected type
      let entityValue = '';
      let entityUrl = attrs?.entityUrl;
      let entityLabel = attrs?.entityLabel;
      
      if (href.startsWith('entity:')) {
        const [entityType] = href.replace('entity:', '').split('/');
        if (entityType === type) {
          entityValue = href;
        } else {
          // Different entity type, clear the values
          entityUrl = undefined;
          entityLabel = undefined;
        }
      }
      
      currentEntitySelectorField = new EntitySelectorField({
        label: profile.label,
        type,
        profile,
        value: entityValue,
        required: true,
        url: entityUrl,
        entityLabel: entityLabel,
        app: appWrapper
      }, appWrapper);
      const field = currentEntitySelectorField.render();
      targetContainer.appendChild(field);
      // Listen to the EntitySelectorField's change event
      field.addEventListener('entity-selector:change', () => {
        validateForm();
      });
    }
  }

  // Initial target field
  updateTargetField(typeField.read(typeSelect) as EntitySelectorProfileType);

  // Update target field when type changes
  typeSelect.addEventListener('change', function () {
    updateTargetField(typeField.read(typeSelect) as EntitySelectorProfileType);
    validateForm(); // Revalidate after field update
  });

  // Buttons
  const cancelResult = renderer.elements.button('Cancel', {
    className: 'button button--secondary',
    listeners: {
      click: (e: Event) => {
        e.preventDefault();
        dialog.close();
      }
    }
  });
  const cancel = cancelResult.element;

  const submitResult = renderer.elements.button(isNew ? 'Insert' : 'Update', {
    className: 'button button--primary',
    disabled: true,
    listeners: {
      click: (e: Event) => {
    e.preventDefault();
    // Validate all fields
    const textValue = (textInput as HTMLInputElement).value;
    const typeValue = (typeSelect as HTMLSelectElement).value;
    const targetInput = targetContainer.querySelector('input') as HTMLInputElement;
    const targetValue = targetInput ? targetInput.value : '';
    const variantValue = variantOptions.length > 1 && variantSelect ? variantSelect.value : 'link';
    if (!textValue || !typeValue || !targetValue) {
      return;
    }
    const tr = view.state.tr;
    const isInternal = typeValue !== 'external';
    const linkType = isInternal ? LinkType.INTERNAL : LinkType.EXTERNAL;
    
    let linkAttrs: any;
    
    if (isInternal) {
      linkAttrs = {
        href: targetValue, // This should be in entity:type/uuid format
        text: textValue,
        linkType,
        variant: variantValue,
        entityUuid: targetInput?.dataset['entityUuid'],
        entityType: targetInput?.dataset['entityType'],
        entityUrl: targetInput?.dataset['entityUrl'],
        entityLabel: targetInput?.dataset['entityLabel'],
        linkUri: null
      };
    } else {
      linkAttrs = {
        href: targetValue,
        text: textValue,
        linkType,
        variant: variantValue,
        entityUuid: null,
        entityType: null,
        entityUrl: null,
        entityLabel: null,
        linkUri: targetValue
      };
    }
    
    tr.insertText(textValue, from, to);
    tr.addMark(
      from,
      from + textValue.length,
      markType.create(linkAttrs)
    );
    view.dispatch(tr);
    dialog.close();
      }
    }
  });
  const submit = submitResult.element;

  // Add validation to enable/disable submit button
  function validateForm() {
    const textValue = (textInput as HTMLInputElement).value;
    const typeValue = (typeSelect as HTMLSelectElement).value;
    const targetInput = targetContainer.querySelector('input') as HTMLInputElement;
    const targetValue = targetInput ? targetInput.value : '';
    if (!textValue || !typeValue || !targetValue) {
      submit.disabled = true;
      return;
    }
    if (typeValue !== 'external') {
      if (!currentEntitySelectorField?.isValid()) {
        submit.disabled = true;
        return;
      }
    }
    submit.disabled = false;
  }

  form.addEventListener('input', validateForm);
  form.addEventListener('change', validateForm);
  validateForm(); // Initial validation

  buttons.appendChild(submit);
  buttons.appendChild(cancel);
  form.appendChild(fields);
  form.appendChild(buttons);

  const dialog = renderer.components.dialog({
    title: isNew ? 'Insert Link' : 'Edit Link',
    content: form,
    width: '400px',
    height: 'auto',
    modal: true
  });
  
  dialog.open();
}
