import { MarkSpec, Mark, MarkType } from "prosemirror-model"
import { EditorView } from "prosemirror-view"
import { Plugin } from "prosemirror-state"
import { createLinkDialog } from "./link-dialog"
import { EntitySelectorPluginSettings } from "./entity-selector"
import { ApplicationWrapper } from "../wrappers/interfaces"
import { findMarkAtPosition } from "../utils"

export enum LinkType {
  INTERNAL = 'internal',
  EXTERNAL = 'external'
}

export interface LinkVariantConfig {
  [key: string]: string;
}

// Helper function to determine if a link is internal
function isInternalLink(href: string): boolean {
  return href.startsWith('entity:') || href.startsWith('internal:');
}

export const linkMarkSpec: MarkSpec = {
  attrs: {
    href: {validate: "string"},
    title: {default: null, validate: "string|null"},
    linkType: {default: LinkType.EXTERNAL, validate: "string"},
    entityUuid: {default: null, validate: "string|null"},
    entityType: {default: null, validate: "string|null"},
    entityUrl: {default: null, validate: "string|null"},
    entityLabel: {default: null, validate: "string|null"},
    linkUri: {default: null, validate: "string|null"},
    variant: {default: 'link', validate: "string"}
  },
  inclusive: false,
  parseDOM: [{tag: "a[href]", getAttrs(dom: HTMLElement) {
    const href = dom.getAttribute("href") || "";
    const isInternal = isInternalLink(href);
    
    // Parse internal link attributes
    if (isInternal) {
      const [type, uuid] = href.replace('entity:', '').split('/');
      return {
        href,
        title: dom.getAttribute("title"),
        linkType: LinkType.INTERNAL,
        entityUuid: uuid,
        entityType: type,
        entityUrl: dom.getAttribute("data-entity-url") || null,
        entityLabel: dom.getAttribute("data-entity-label") || null,
        linkUri: null,
        variant: dom.getAttribute("data-variant") || 'link'
      };
    }
    
    // External link attributes
    return {
      href,
      title: dom.getAttribute("title"),
      linkType: LinkType.EXTERNAL,
      entityUuid: null,
      entityType: null,
      entityUrl: null,
      entityLabel: null,
      linkUri: href,
      variant: dom.getAttribute("data-variant") || 'link',
    };
  }}],
  toDOM(node) { 
    const {href, title, linkType, entityUuid, entityType, entityUrl, entityLabel, linkUri, variant} = node.attrs;
    const attrs: Record<string, string> = {
      href,
      class: `prosemirror-link ${linkType === LinkType.INTERNAL ? 'internal' : ''} ${variant !== 'link' ? variant : ''}`,
      'data-variant': variant
    };
    
    if (title) attrs.title = title;
    
    if (linkType === LinkType.INTERNAL) {
      attrs['data-entity-url'] = entityUrl || '';
      attrs['data-entity-label'] = entityLabel || '';
    }
    
    return ["a", attrs, 0];
  }
} satisfies MarkSpec;

interface LinkTooltipOptions {
  entitySelectorSettings: EntitySelectorPluginSettings;
  variants: LinkVariantConfig;
  appWrapper: ApplicationWrapper;
}

class LinkTooltip {
  private tooltip: HTMLElement;
  private activeLink: { mark: Mark, from: number, to: number } | null = null;
  private entitySelectorSettings: EntitySelectorPluginSettings;

  constructor(private view: EditorView, private options: LinkTooltipOptions) {
    this.entitySelectorSettings = options.entitySelectorSettings;
    
    // Create tooltip using renderer
    const renderer = options.appWrapper.getRendererFactory();
    const tooltipResult = renderer.components.tooltip({
      content: '',
      className: 'prosemirror-tooltip prosemirror-link-tooltip',
      interactive: true
    });
    
    this.tooltip = tooltipResult.element;
    
    // Append tooltip to body
    document.body.appendChild(this.tooltip);

    this.handleClick = this.handleClick.bind(this);
    this.handleDocumentClick = this.handleDocumentClick.bind(this);

    this.view.dom.addEventListener('click', this.handleClick);
    this.options.appWrapper.addDocumentEventListener('click', this.handleDocumentClick);
  }

  private handleClick(event: MouseEvent) {
    const target = event.target as HTMLElement;
    const link = target.closest('a');
    
    if (link) {
      const pos = this.view.posAtDOM(link, 0);
      
      if (pos === null || pos < 0) {
        console.warn('Could not determine position for link element');
        return;
      }
      
      const linkMarkResult = findMarkAtPosition(this.view, pos, this.view.state.schema.marks.link);
      
      if (linkMarkResult) {
        const rect = link.getBoundingClientRect();
        this.showTooltip(rect, linkMarkResult.mark, linkMarkResult.from, linkMarkResult.to);
        event.preventDefault();
      }
    }
  }

  private handleDocumentClick(event: Event) {
    if (!this.tooltip.contains(event.target as Node) && !(event.target as HTMLElement).closest('a')) {
      this.hideTooltip();
    }
  }

  private sanitizeUrl(url: string): string {
    // Prevent javascript: and data: URLs which could be used for XSS
    const dangerousProtocols = ['javascript:', 'data:', 'vbscript:', 'file:', 'about:'];
    const lowerUrl = url.toLowerCase().trim();
    
    for (const protocol of dangerousProtocols) {
      if (lowerUrl.startsWith(protocol)) {
        return '#'; // Return safe fallback
      }
    }
    
    return url;
  }

  private showTooltip(rect: DOMRect, mark: Mark, from: number, to: number) {
    this.activeLink = { mark, from, to };
    
    const href = mark.attrs.href;
    const label = mark.attrs.entityLabel || href;
    const url = mark.attrs.entityUrl || href;
    
    // Safely create tooltip content using DOM manipulation to prevent XSS
    const renderer = this.options.appWrapper.getRendererFactory();
    
    const contentResult = renderer.elements.div({ className: 'prosemirror-link-tooltip-content' });
    const content = contentResult.element;
    
    // Create link element safely
    const link = document.createElement('a');
    link.textContent = label;
    link.href = this.sanitizeUrl(url);
    link.target = '_blank';
    link.title = url;
    link.className = 'prosemirror-link-tooltip-link';
    
    const editButtonResult = renderer.elements.button('Edit', {
      className: 'prosemirror-link-tooltip-button edit-link'
    });
    const editButton = editButtonResult.element;
    
    const unlinkButtonResult = renderer.elements.button('Unlink', {
      className: 'prosemirror-link-tooltip-button unlink'
    });
    const unlinkButton = unlinkButtonResult.element;
    
    content.appendChild(link);
    content.appendChild(editButton);
    content.appendChild(unlinkButton);
    
    // Clear and set tooltip content
    this.tooltip.innerText = '';
    this.tooltip.appendChild(content);

    this.tooltip.classList.remove('hidden');
    
    // Calculate position with scroll offset for absolute positioning
    const scrollX = window.pageXOffset || document.documentElement.scrollLeft;
    const scrollY = window.pageYOffset || document.documentElement.scrollTop;
    
    const tooltipRect = this.tooltip.getBoundingClientRect();
    let left = rect.left + scrollX;
    let top = rect.bottom + scrollY + 5;

    // Ensure tooltip stays within viewport
    if (rect.left + tooltipRect.width > window.innerWidth) {
      left = window.innerWidth - tooltipRect.width - 10 + scrollX;
    }
    if (rect.bottom + tooltipRect.height + 5 > window.innerHeight) {
      top = rect.top + scrollY - tooltipRect.height - 5;
    }

    this.tooltip.style.setProperty('--tooltip-left', `${left}px`);
    this.tooltip.style.setProperty('--tooltip-top', `${top}px`);

    unlinkButton.addEventListener('click', () => {
      if (this.activeLink) {
        const tr = this.view.state.tr.removeMark(
          this.activeLink.from,
          this.activeLink.to,
          this.view.state.schema.marks.link
        );
        this.view.dispatch(tr);
        this.hideTooltip();
      }
    });

    editButton.addEventListener('click', () => {
      if (this.activeLink) {
        const text = this.view.state.doc.textBetween(this.activeLink.from, this.activeLink.to);
        createLinkDialog({
          view: this.view,
          markType: this.view.state.schema.marks.link,
          entitySelectorSettings: this.entitySelectorSettings,
          isNew: false,
          from: this.activeLink.from,
          to: this.activeLink.to,
          attrs: {
            href: this.activeLink.mark.attrs.href,
            linkType: this.activeLink.mark.attrs.linkType,
            text,
            ...this.activeLink.mark.attrs
          },
          variants: this.options.variants,
          appWrapper: this.options.appWrapper
        });
      }
    });
  }

  private hideTooltip() {
    this.tooltip.classList.add('hidden');
    this.activeLink = null;
  }

  destroy() {
    this.tooltip.remove();
    this.view.dom.removeEventListener('click', this.handleClick);
    this.options.appWrapper.removeDocumentEventListener('click', this.handleDocumentClick);
  }
}

export function createLinkPlugin(entitySelectorSettings: EntitySelectorPluginSettings, variants: LinkVariantConfig, appWrapper: ApplicationWrapper) {
  return new Plugin({
    view(editorView) {
      return new LinkTooltip(editorView, { entitySelectorSettings, variants, appWrapper });
    }
  });
}
