import {DOMSerializer, Node} from "prosemirror-model"
import { initializeProsemirror, ProseMirrorInstance } from "./wrappers/init";
import { ProsemirrorSettings, ProsemirrorComponent } from "./wrappers/config";
import { DrupalWrapper } from "./wrappers/drupal-wrapper";
import { createListBlock } from "./core/list-block";
import { createLeafBlock } from "./core/leaf-block";
import { baseNodeTemplate } from "./components/base-node";
import { ProseMirrorAPI } from "./extensibility/api";

/**
 * @file
 *
 * Prosemirror implementation of {@link Drupal.editors} API.
 */

((Drupal, debounce, $, once) => {
  const instances = new Map<HTMLElement, ProseMirrorInstance>();

  /**
   * The Prosemirror instances.
   *
   * @type {Map}
   */
  Drupal.ProsemirrorInstances = instances;

  /**
   * The callback functions.
   *
   * @type {Map}
   */
  const callbacks = new Map();

  Drupal.ProsemirrorTemplateTypes = Object.assign({
    list_block: createListBlock,
    leaf_block: createLeafBlock,
    base_node: baseNodeTemplate,
  }, Drupal.ProsemirrorTemplateTypes ?? {});

  /**
   * Extensibility API for ProseMirror
   *
   * This provides access to all ProseMirror modules and allows other modules
   * to register their own nodes, marks, plugins, and menu extensions.
   */
  Drupal.ProseMirrorAPI = ProseMirrorAPI;

  /**
   * Public API for Drupal ProseMirror integration.
   *
   * @namespace
   */
  Drupal.prosemirror = {
    /**
     * Variable storing the current dialog's save callback.
     *
     * @type {?function}
     */
    saveCallback: null,

    /**
     * Open a dialog for a Drupal-based plugin.
     *
     * This dynamically loads jQuery UI (if necessary) using the Drupal AJAX
     * framework, then opens a dialog at the specified Drupal path.
     *
     * @param {string} url
     *   The URL that contains the contents of the dialog.
     * @param {function} saveCallback
     *   A function to be called upon saving the dialog.
     * @param {object} dialogSettings
     *   An object containing settings to be passed to the jQuery UI.
     */
    openDialog(url: string, saveCallback: (data: any) => void, dialogSettings: Record<string, any>) {
      // Add a consistent dialog class.
      const classes = dialogSettings.dialogClass
        ? dialogSettings.dialogClass.split(' ')
        : [];
      //classes.push('ui-dialog--narrow');
      dialogSettings.dialogClass = classes.join(' ');
      dialogSettings.autoResize =
        window.matchMedia('(min-width: 600px)').matches;
      dialogSettings.width = 'auto';

      const ajaxDialog = Drupal.ajax({
        dialog: dialogSettings,
        dialogType: 'modal',
        selector: '.prosemirror-dialog-loading-link',
        url,
        progress: { type: 'fullscreen' },
        submit: {
          editor_object: {},
        },
      });
      ajaxDialog.execute();

      // Store the save callback to be executed when this dialog is closed.
      Drupal.prosemirror.saveCallback = saveCallback;
    },
  };

  /**
   * Integration of Prosemirror with the Drupal editor API.
   *
   * @namespace
   *
   * @see Drupal.editorAttach
   */
  Drupal.editors.prosemirror = {
    /**
     * Editor attach callback.
     *
     * @param {HTMLElement} element
     *   The element to attach the editor to.
     * @param {object} format
     *   The text format for the editor.
     */
    attach(element: HTMLTextAreaElement, format: Record<string, any>) {
      const jElement = $(element);

      const { config } = format.editorSettings;

      //console.debug("ProseMirror attach", format);

      const { metadataUrl, previewURL, previewCsrfToken } = config.plugins.media.drupalMedia;
      Drupal.prosemirror.mediaMetadataUrl = metadataUrl;
      Drupal.prosemirror.mediaPreviewCsrfToken = previewCsrfToken;
      Drupal.prosemirror.mediaPreviewUrl = previewURL;

      Drupal.prosemirror.mediaLibraryUrl = config.plugins.media.drupalMedia.libraryURL;

      const entitySelectorPluginSettings = config.plugins.entitySelector;

      // Get components configuration from server settings and convert to array
      const componentsConfig = config.components || {};
      const components: ProsemirrorComponent[] = Object.entries(componentsConfig).map(([name, component]: [string, any]) => ({
        ...component,
        options: {
          ...component.options,
          name
        }
      }));

      // Get system nodes configuration
      const systemNodes = config.systemNodes || {};

      // Get marks configuration
      const marks = config.marks || {};

      const updateValue = debounce((data: string) => jElement.val(data), 400);

      const onChange = (doc: Node) => {
        updateValue(JSON.stringify(doc.toJSON()));

        // Check for listeners
        const callback = callbacks.get(element);
        if (callback) {
          callback();
        }
      };

      const prosemirrorSettings: ProsemirrorSettings = {
        ...format.editorSettings.prosemirrorSettings,
        templateTypes: Drupal.ProsemirrorTemplateTypes,
        components,
        systemNodes,
        marks,
        media: {
          drupalMedia: {
            libraryURL: config.plugins.media.drupalMedia.libraryURL,
            metadataUrl,
            previewURL,
            previewCsrfToken,
            openDialog: Drupal.prosemirror.openDialog,
            dialogSettings: config.plugins.media.drupalMedia.dialogSettings,
          }
        },
        entitySelector: entitySelectorPluginSettings,
        onChange,
      };

      const appWrapper = new DrupalWrapper();

      const instance = initializeProsemirror(element, prosemirrorSettings, appWrapper);

      //console.debug("ProseMirror instantiated", instance, format);

      const embedMenuItem = instance.editableElement.querySelector(".ProseMirror-menu-dropdown.menu-embed");
      let embedMenuWrapper: HTMLElement|null = embedMenuItem?.parentElement ?? null;
      while(embedMenuWrapper && !embedMenuWrapper.classList.contains("ProseMirror-menuitem")) {
        embedMenuWrapper = embedMenuWrapper.parentElement;
      }
      if(embedMenuWrapper) {
        embedMenuWrapper.classList.add("pull-menu-item-right");
        embedMenuWrapper.classList.add("highlight-menu-item");
      }

      $(element).hide();

      // Save a reference to the initialized instance.
      instances.set(element, instance);

      // Set the initial value of the editor. This is to avoid HTML-escaping the JSON when the editor is not used.
      // Without this, content will look like {&quot;...&quot;} which is not valid JSON.
      // Drupal won't let us correct this without the timeout.
      setTimeout(() => {
        onChange(instance.view.state.doc);

        //console.debug("ProseMirror init done", instance);
      }, 100);
    },

    /**
     * Editor detach callback.
     *
     * @param {HTMLElement} element
     *   The element to detach the editor from.
     * @param {string} format
     *   The text format used for the editor.
     * @param {string} trigger
     *   The event trigger for the detach.
     */
    detach(element: HTMLTextAreaElement, format: string, trigger: string) {
      const instance = instances.get(element);
      if (!instance) {
        return;
      }

      const {editableElement, view} = instance;

      const jElement = $(element);

      if (trigger === 'serialize') {
        jElement.val(JSON.stringify(view.state.doc.toJSON()));
        //console.debug("ProseMirror serialize", jElement.val());
      } else {
        const serializer = DOMSerializer.fromSchema(view.state.schema);
        const node = serializer.serializeFragment(view.state.doc.content);

        jElement.val($('<div />').append(node).html());
        //console.debug("ProseMirror serialize", jElement.val());

        jElement.show();
        editableElement.remove();

        instances.delete(element);
        callbacks.delete(element);
      }
    },

    /**
     * Registers a callback which Prosemirror will call when data changes.
     *
     * @param {HTMLElement} element
     *   The element where the change occurred.
     * @param {function} callback
     *   Callback called with the value of the editor.
     */
    onChange(element: HTMLTextAreaElement, callback: () => void) {
      callbacks.set(element, debounce(callback, 400, true));
    },
  };

  // Respond to dialogs that are saved, sending data back to the editor.
  $(window).on('editor:dialogsave', (e: Event, values: any) => {
    if (Drupal.prosemirror.saveCallback) {
      Drupal.prosemirror.saveCallback(values);
    }
  });

  // Respond to dialogs that are closed, removing the current save handler.
  window.addEventListener('dialog:afterclose', () => {
    if (Drupal.prosemirror.saveCallback) {
      Drupal.prosemirror.saveCallback = null;
    }
  });

// @ts-ignore
})(Drupal, Drupal.debounce, jQuery, once);
