import { BaseElementRenderer } from '../base';
import { 
  ElementAttributes, 
  InputAttributes, 
  SelectOption, 
  ButtonAttributes,
  RenderResult 
} from '../interfaces';

export class DOMElementRenderer extends BaseElementRenderer {
  div(attrs?: ElementAttributes): RenderResult<HTMLDivElement> {
    const element = document.createElement('div');
    this.applyAttributes(element, attrs);
    return this.createRenderResult(element, attrs?.listeners);
  }
  
  span(attrs?: ElementAttributes): RenderResult<HTMLSpanElement> {
    const element = document.createElement('span');
    this.applyAttributes(element, attrs);
    return this.createRenderResult(element, attrs?.listeners);
  }
  
  input(attrs?: InputAttributes): RenderResult<HTMLInputElement> {
    const element = document.createElement('input');
    
    if (attrs) {
      if (attrs.type) element.type = attrs.type;
      if (attrs.placeholder) element.placeholder = attrs.placeholder;
      if (attrs.value) element.value = attrs.value;
      if (attrs.required !== undefined) element.required = attrs.required;
      if (attrs.name) element.name = attrs.name;
      if (attrs.autocomplete) element.setAttribute('autocomplete', attrs.autocomplete);
    }
    
    this.applyAttributes(element, attrs);
    return this.createRenderResult(element, attrs?.listeners);
  }
  
  button(text: string, attrs?: ButtonAttributes): RenderResult<HTMLButtonElement> {
    const element = document.createElement('button');
    element.textContent = text;
    
    if (attrs) {
      if (attrs.disabled !== undefined) element.disabled = attrs.disabled;
    }
    
    this.applyAttributes(element, attrs);
    return this.createRenderResult(element, attrs?.listeners);
  }
  
  label(text: string, attrs?: ElementAttributes & { for?: string }): RenderResult<HTMLLabelElement> {
    const element = document.createElement('label');
    element.textContent = text;
    element.classList.add('form-label');
    
    if (attrs?.for) {
      element.setAttribute('for', attrs.for);
    }
    
    this.applyAttributes(element, attrs);
    return this.createRenderResult(element, attrs?.listeners);
  }
  
  select(options: SelectOption[], attrs?: ElementAttributes): RenderResult<HTMLSelectElement> {
    const element = document.createElement('select');
    
    options.forEach(option => {
      const optionElement = document.createElement('option');
      optionElement.value = option.value;
      optionElement.textContent = option.label;
      if (option.selected) {
        optionElement.selected = true;
      }
      element.appendChild(optionElement);
    });
    
    this.applyAttributes(element, attrs);
    return this.createRenderResult(element, attrs?.listeners);
  }
  
  form(attrs?: ElementAttributes): RenderResult<HTMLFormElement> {
    const element = document.createElement('form');
    this.applyAttributes(element, attrs);
    // Prevent submission of the form as that would break the editor.
    element.addEventListener('submit', (e) => {
      e.preventDefault();
    });
    return this.createRenderResult(element, attrs?.listeners);
  }
}
