import { ApplicationWrapper, MediaAttributes } from './interfaces';
import { RendererFactory } from '../renderers/interfaces';
import { DOMRendererFactory } from '../renderers/dom-renderer-factory';

declare const Drupal: any;
declare const jQuery: any;

/**
 * Drupal-specific implementation of ApplicationWrapper
 */
export class DrupalWrapper implements ApplicationWrapper {
  private rendererFactory: RendererFactory = new DOMRendererFactory();
  dialogSettings = {
    dialogClass: '',
    autoResize: window.matchMedia('(min-width: 600px)').matches,
    width: 'auto'
  };

  selectMedia(callback: (attributes: MediaAttributes) => void): void {
    const mediaLibraryUrl = Drupal.prosemirror.mediaLibraryUrl;
    if (!mediaLibraryUrl) {
      console.error('Media library URL not configured');
      return;
    }

    this.openModalDialog(mediaLibraryUrl, (data: any) => {
      if (data && data.attributes) {
        callback({
          'data-entity-type': data.attributes['data-entity-type'],
          'data-entity-uuid': data.attributes['data-entity-uuid'],
          updatedAt: Date.now()
        });
      }
    }, this.dialogSettings);
  }

  openModalDialog(url: string, saveCallback: (data: any) => void, dialogSettings: Record<string, any>) {
    // Add a consistent dialog class.
    const classes = dialogSettings.dialogClass
      ? dialogSettings.dialogClass.split(' ')
      : [];
    dialogSettings.dialogClass = classes.join(' ');
    dialogSettings.autoResize =
      window.matchMedia('(min-width: 600px)').matches;
    dialogSettings.width = 'auto';

    const ajaxDialog = Drupal.ajax({
      dialog: dialogSettings,
      dialogType: 'modal',
      selector: '.prosemirror-dialog-loading-link',
      url,
      progress: { type: 'fullscreen' },
      submit: {
        editor_object: {},
      },
    });
    ajaxDialog.execute();

    // Store the save callback to be executed when this dialog is closed.
    Drupal.prosemirror.saveCallback = saveCallback;
  }

  openCustomDialog(content: HTMLElement, saveCallback: (data: any) => void, dialogSettings: Record<string, any>) {
    const dialog = Drupal.dialog(content, {
      ...dialogSettings,
      close: () => {
        if (saveCallback) {
          saveCallback({});
        }
      }
    });

    dialog.showModal();

    return {
      close: () => {
        dialog.close();
      }
    };
  }

  ajax(options: {
    url: string;
    type?: string;
    data?: any;
    dataType?: string;
    success?: (response: any) => void;
    error?: (error: any) => void;
    headers?: Record<string, string>;
  }): void {
    jQuery.ajax({
        ...options,
        dataType: options.dataType || 'text',
        jsonp: false,
    });
  }

  getMediaPreviewUrl(): string {
    return Drupal.prosemirror.mediaPreviewUrl;
  }

  getMediaPreviewCsrfToken(): string {
    return Drupal.prosemirror.mediaPreviewCsrfToken;
  }

  getMediaMetadataUrl(): string {
    return Drupal.prosemirror.mediaMetadataUrl;
  }

  addWindowEventListener(event: string, callback: (event: Event, ...args: any[]) => void): void {
    jQuery(window).on(event, callback);
  }

  removeWindowEventListener(event: string, callback: (event: Event, ...args: any[]) => void): void {
    jQuery(window).off(event, callback);
  }

  addDocumentEventListener(event: string, callback: (event: Event, ...args: any[]) => void): void {
    jQuery(document).on(event, callback);
  }

  removeDocumentEventListener(event: string, callback: (event: Event, ...args: any[]) => void): void {
    jQuery(document).off(event, callback);
  }

  parseHTML(html: string): HTMLElement {
    return jQuery(html)[0];
  }

  getElementValue(element: HTMLElement): string {
    const jElement = jQuery(element);

    // Drupal's filter_html filter not only restricts the RENDERED HTML but also the EDITED HTML.
    // This results in wrongly escaped characters as it not only limits HTML tags/attributes but also
    // HTML-escapes content even though we need them as plain text.
    // This is fine to do when rendering, but breaks the editor.
    // To avoid this incorrect escaping but still allowing to use this filter, we HTML-escape *everything*
    // and then HTML-unescape *everything* in the frontend before loading the editor.
    return jElement.html(jElement.val()).text();
  }

  setElementValue(element: HTMLElement, value: string): void {
    jQuery(element).val(value);
  }

  async getMediaPreview(uuid: string): Promise<{ preview: string, label: string | null }> {
    return new Promise((resolve, reject) => {
      this.ajax({
        url: this.getMediaPreviewUrl(),
        type: 'GET',
        data: {
          text: `<drupal-media data-entity-type="media" data-entity-uuid="${uuid}" />`,
          uuid: uuid
        },
        headers: {
          'X-Drupal-MediaPreview-CSRF-Token': this.getMediaPreviewCsrfToken()
        },
        success: (response: any) => {
          const label = response.headers?.['drupal-media-label'] || null;
          let preview = response.data || response;
          preview = preview.replace(/\s+/g, " ").replace(/<!--.*?-->/g, "").trim();
          resolve({ preview, label });
        },
        error: (error: any) => {
          reject(new Error(`Failed to load media preview: ${error.status}`));
        }
      });
    });
  }

  getRendererFactory(): RendererFactory {
    return this.rendererFactory;
  }
}
