import { DOMParser, Node, Schema, MarkSpec, DOMOutputSpec, NodeSpec } from "prosemirror-model";
import { EditorState, EditorStateConfig, Plugin } from "prosemirror-state";
import { EditorView } from "prosemirror-view";
import { history } from "prosemirror-history";
import { search } from "prosemirror-search";
import { goToNextCell, tableEditing, tableNodes } from "prosemirror-tables";
import { keymap } from "prosemirror-keymap";
import { dropCursor } from "prosemirror-dropcursor";
import { gapCursor } from "prosemirror-gapcursor";
import { baseKeymap } from "prosemirror-commands";
import { addListNodes, liftListItem, sinkListItem } from "prosemirror-schema-list";
import { menuBar, MenuItem } from "prosemirror-menu";
import { buildInputRules } from "../import/inputrules";
import { buildKeymap } from "../import/keymap";
import { buildMenuItems, canInsert } from "../import/menu";
import { proseMirrorPluginRegistry } from "../extensibility/plugin-registry";
import { MediaView, mediaNodeSpec } from "../plugins/media";
import { createCodeBlockComponents } from "../plugins/code-block";
import { linkMarkSpec, createLinkPlugin } from "../plugins/link";
import { createTablePlugin } from "../plugins/table";
import { ApplicationWrapper } from "./interfaces";
import { ProsemirrorSettings } from "./config";



export interface ProseMirrorInstance {
  view: EditorView;
  editableElement: HTMLElement;
}

export function initializeProsemirror(
  element: HTMLTextAreaElement,
  settings: ProsemirrorSettings,
  appWrapper: ApplicationWrapper
): ProseMirrorInstance {
  const components = settings.components.map((component) => {
    const templateType = settings.templateTypes[component.type];
    if(!templateType) {
      throw new Error(`Template type ${component.type} not found`);
    }
    const componentDef = templateType(component.options, appWrapper);
    // Store the original component type for filtering
    (componentDef as any).componentType = component.type;
    return componentDef;
  });

  // Get system nodes configuration
  const systemNodesConfig = settings.systemNodes || {};

  // Create code block component with language settings
  const codeBlockSettings = systemNodesConfig.code_block || {};
  const codeBlockComponent = createCodeBlockComponents(appWrapper)({
    languages: codeBlockSettings.languages || {},
  });

  // Build system nodes from configuration
  const systemNodes: Record<string, NodeSpec> = {};
  let defaultElementName: string | undefined;

  if (systemNodesConfig) {
    for (const [name, spec] of Object.entries(systemNodesConfig)) {
      if (name === 'defaultElement') {
        defaultElementName = spec as string;
      } else {
        systemNodes[name] = spec as NodeSpec;
      }
    }
  }

  // Build component specs map
  const componentSpecs: Record<string, NodeSpec> = components.reduce((acc, component) => {
    acc[component.name] = component.spec;
    return acc;
  }, {} as Record<string, NodeSpec>);

  // Build nodes in specific order: doc, default element, then others
  const baseNodes: Record<string, NodeSpec> = {};

  // 1. Add doc node first
  if (systemNodes.doc) {
    baseNodes.doc = systemNodes.doc;
  }

  // 2. Add default element second (if specified)
  if (defaultElementName && componentSpecs[defaultElementName]) {
    baseNodes[defaultElementName] = componentSpecs[defaultElementName];
    delete componentSpecs[defaultElementName]; // Remove from componentSpecs to avoid duplication
  }

  // 3. Add text node
  if (systemNodes.text) {
    baseNodes.text = systemNodes.text;
  }

  // 4. Add system nodes based on configuration

  // Add media if enabled
  if (systemNodesConfig.media) {
    baseNodes.media = mediaNodeSpec;
  }

  // Icons are now provided by extension modules (e.g., prosemirror_fontawesome_icons)

  // Add code block if enabled
  if (systemNodesConfig.code_block) {
    baseNodes.code_block = codeBlockComponent.spec;
  }

  // Add table nodes if enabled
  if (systemNodesConfig.table) {
    // Table nodes will be added via prosemirror-tables
  }

  // Add list nodes if enabled
  if (systemNodesConfig.bullet_list) {
    // List nodes will be added via prosemirror-schema-list
  }

  if (systemNodesConfig.ordered_list) {
    // List nodes will be added via prosemirror-schema-list
  }

  // Add heading nodes if enabled
  if (systemNodesConfig.heading) {
    // Heading nodes are configured as individual elements, not system nodes
    // They will be added via the components system
  }

  // Add all other component nodes
  Object.assign(baseNodes, componentSpecs);

  //console.debug('ProseMirror Init: Base nodes:', baseNodes);

  // Add registered node plugins from extension registry
  const registeredNodePlugins = proseMirrorPluginRegistry.getNodePlugins();
  //console.debug('ProseMirror Init: Found', registeredNodePlugins.size, 'registered node plugins');
  Array.from(registeredNodePlugins.entries()).forEach(([name, plugin]) => {
    //console.debug('ProseMirror Init: Adding registered node plugin:', name);
    baseNodes[name] = plugin.spec;
  });

  // Convert marks from Drupal configuration
  const drupalMarks: Record<string, MarkSpec> = {};

  if (settings.marks) {
    for (const [name, markConfig] of Object.entries(settings.marks)) {
      const markSpec: MarkSpec = {};

      // Convert parseDOM rules
      if (markConfig.parse_dom && markConfig.parse_dom.length > 0) {
        markSpec.parseDOM = markConfig.parse_dom.map(rule => {
          const parseDOMRule: any = {};

          if (rule.tag) {
            parseDOMRule.tag = rule.tag;
          }
          if (rule.style) {
            parseDOMRule.style = rule.style;
          }
          if (rule.get_attrs) {
            // Handle special cases for getAttrs functions
            if (rule.tag === 'b' && rule.get_attrs === 'font-weight-normal') {
              parseDOMRule.getAttrs = (node: HTMLElement) => node.style.fontWeight != "normal" && null;
            } else if (rule.style === 'font-weight' && rule.get_attrs === 'bold-regex') {
              parseDOMRule.getAttrs = (value: string) => /^(bold(er)?|[5-9]\d{2,})$/.test(value) && null;
            }
          }
          if (rule.clear_mark) {
            parseDOMRule.clearMark = (m: any) => m.type.name == rule.clear_mark;
          }

          return parseDOMRule;
        });
      }

      // Convert toDOM specification
      if (markConfig.to_dom) {
        const toDomTag = markConfig.to_dom.tag || 'span';
        const toDomAttrs = markConfig.to_dom.attrs || {};

        markSpec.toDOM = () => {
          if (Object.keys(toDomAttrs).length > 0) {
            return [toDomTag, toDomAttrs, 0] as DOMOutputSpec;
          } else {
            return [toDomTag, 0] as DOMOutputSpec;
          }
        };
      }

      // Add additional attributes
      if (markConfig.attributes) {
        Object.assign(markSpec, markConfig.attributes);
      }

      drupalMarks[name] = markSpec;
    }
  }

  // Add registered mark plugins from extension registry
  const registeredMarkPlugins = proseMirrorPluginRegistry.getMarkPlugins();
  Array.from(registeredMarkPlugins.entries()).forEach(([name, plugin]) => {
    drupalMarks[name] = plugin.spec;
  });

  // Always ensure we have the link mark
  const baseMarks = {
    ...drupalMarks,
    link: linkMarkSpec,
  };

  const baseSchema = new Schema({
    nodes: baseNodes,
    marks: baseMarks,
  });

  // Build the final schema with optional features
  let finalNodes = baseSchema.spec.nodes;

  // Add lists if enabled
  if (systemNodesConfig.bullet_list || systemNodesConfig.ordered_list) {
    const lists = [
      systemNodesConfig.ordered_list ? 'ordered_list' : '',
      systemNodesConfig.bullet_list ? 'bullet_list' : '',
    ].filter(Boolean);
    finalNodes = addListNodes(finalNodes, `paragraph (${lists.join(' | ')})*`, "block");
  }

  // Add tables if enabled
  if (systemNodesConfig.table) {
    const tableSettings = systemNodesConfig.table;
    finalNodes = finalNodes.append(
      tableNodes({
        tableGroup: 'block nested',
        cellContent: 'block+',
        cellAttributes: {
          // TODO: Add background color attribute if needed. But should probably use variants instead to be consistent with the rest of the system.
          /*background: {
            default: null,
            getFromDOM(dom) {
              return dom.style.backgroundColor || null;
            },
            setDOMAttr(value, attrs) {
              if (value)
                attrs.style = (attrs.style || '') + `background-color: ${value};`;
            },
          },*/
        },
      })
    );
  }

  const useSchema = settings.schema ?? new Schema({
    nodes: finalNodes,
    marks: baseSchema.spec.marks,
  });

  let plugins: Plugin<any>[] = [
    buildInputRules(useSchema),
    keymap(buildKeymap(useSchema, settings.mapKeys)),
  ];

  // Add table plugins if tables are enabled
  if (systemNodesConfig.table) {
    plugins.push(keymap({
      Tab: goToNextCell(1),
      'Shift-Tab': goToNextCell(-1),
    }));
    plugins.push(tableEditing());
    plugins.push(createTablePlugin(appWrapper));
  }

  // Add list plugins if lists are enabled
  if (systemNodesConfig.bullet_list || systemNodesConfig.ordered_list) {
    plugins.push(keymap({
      Tab: sinkListItem(useSchema.nodes.list_item),
      'Shift-Tab': liftListItem(useSchema.nodes.list_item),
    }));
  }

  // Add code block keymap if code blocks are enabled
  if (systemNodesConfig.code_block) {
    plugins.push(keymap({
      // When hitting enter and being in the code block, we want to insert a new line within the code block.
      'Enter': (state, dispatch, view) => {
        const { selection } = state;
        const { $from } = selection;
        const node = $from.node();

        if(node.type.name === 'code_block') {
          dispatch?.(state.tr.insertText('\n'));
          return true;
        }
        return false;
      },
    }));
  }

  plugins.push(dropCursor());
  plugins.push(gapCursor());

  // Fall back to default keymap; must be at the end.
  plugins.push(keymap(baseKeymap));

  // Get the root group from system nodes (if available)
  let rootGroup = 'block';
  if (settings.systemNodes?.doc && typeof settings.systemNodes.doc.content === 'string') {
    rootGroup = settings.systemNodes.doc.content.replace(/[\+\*\{\d,\}]/g, '').trim() || 'block';
  }

  // Build insert menu items
  const insertMenuItems: MenuItem[] = [];

  // Add media menu item if enabled
  if (systemNodesConfig.media && useSchema.nodes.media) {
    insertMenuItems.push(new MenuItem({
      title: "Insert media",
      label: "Media",
      enable(state) { return canInsert(state, useSchema.nodes.media) },
      run(state, dispatch) {
        appWrapper.selectMedia((attributes) => {
          dispatch(state.tr.replaceSelectionWith(useSchema.nodes.media.create(attributes)))
        });
      }
    }));
  }

  // Icon menus are now added via registered extensions

  // Add code block menu if enabled
  if (systemNodesConfig.code_block) {
    insertMenuItems.push(codeBlockComponent.menuItem(useSchema));
  }

  //console.debug('ProseMirror Init: Insert menu items:', insertMenuItems, components, useSchema);

  // Add component inserts
  insertMenuItems.push(...components
    .filter(component => {
      // Exclude base_node types from embed menu (they're in other menu sections)
      if ((component as any).componentType === 'base_node') return false;

      // Filter components based on whether they belong to the root group
      const nodeSpec = useSchema.nodes[component.name];
      if (!nodeSpec) return false;

      if(component.menuName === 'insert') {
        return true;
      }

      // Check if the node belongs to the root group
      const groups = nodeSpec.spec.group?.split(' ') || [];
      return groups.includes(rootGroup) || groups.some(g => g.includes('block'));
    })
    .map((component) => component.menuItem(useSchema))
  );

  // Add registered menu extensions for 'insert' section
  const insertMenuExtensions = proseMirrorPluginRegistry.getMenuExtensions('insert');
  insertMenuExtensions.forEach(extension => {
    insertMenuItems.push(extension.menuItem(useSchema, appWrapper));
  });

  // Add registered node plugin menu items
  //console.debug('ProseMirror Init: Adding menu items from registered node plugins...');
  Array.from(registeredNodePlugins.values()).forEach(plugin => {
    if (plugin.menuItem) {
      //console.debug('ProseMirror Init: Adding menu item for plugin:', plugin.name);
      insertMenuItems.push(plugin.menuItem(useSchema, appWrapper));
    } else {
      //console.debug('ProseMirror Init: Plugin has no menu item:', plugin.name);
    }
  });
  //console.debug('ProseMirror Init: Total insert menu items after registered plugins:', insertMenuItems.length);

  // Get link variants from marks configuration
  const linkMarkConfig = settings.marks?.link;
  const linkVariants = linkMarkConfig?.variants || {};

  const menuItems = buildMenuItems(useSchema, insertMenuItems, {
    entitySelector: settings.entitySelector ?? {
      profiles: {
        node: { profile: '', autocompleteUrl: '', label: 'Content' },
        media: { profile: '', autocompleteUrl: '', label: 'Media' }
      }
    },
    rootGroup,
    systemNodes: systemNodesConfig, // Pass system nodes config to menu builder
  }, linkVariants, appWrapper);

  plugins.push(menuBar({
    floating: settings.floatingMenu !== false,
    content: settings.menuContent || menuItems.fullMenu,
  }))

  plugins.push(keymap({
    'Mod-k': (state, dispatch, view) => {
      if(dispatch && view) {
        menuItems.toggleLink?.spec.run(state, dispatch, view, new Event("click"));
        return true;
      }
      return false;
    },
  }));

  if(settings.search !== false) {
    plugins.push(search());
  }

  if (settings.history !== false) {
    plugins.push(history());
  }

  // Icon plugins are now added via registered extensions

  // Add registered plugin extensions
  const registeredPluginExtensions = proseMirrorPluginRegistry.getPluginExtensions();
  Array.from(registeredPluginExtensions.values()).forEach(extension => {
    plugins.push(extension.plugin(useSchema, appWrapper));
  });

  plugins.push(...components.map((component) => component.plugin));

  // Add code block plugin if enabled
  if (systemNodesConfig.code_block) {
    plugins.push(codeBlockComponent.plugin);
  }

  plugins.push(createLinkPlugin(settings.entitySelector ?? {
    profiles: {
      node: { profile: '', autocompleteUrl: '', label: 'Content' },
      media: { profile: '', autocompleteUrl: '', label: 'Media' }
    }
  }, linkVariants, appWrapper));

  plugins = plugins.concat(new Plugin({
    props: {
      attributes: {class: "ProseMirror-Drupal"}
    }
  }))

  const value = appWrapper.getElementValue(element).trim();

  const stateConfig = {
    schema: useSchema,
    plugins
  } satisfies EditorStateConfig;

  let doc: Node|undefined = undefined;

  if(value) {
    try {
      if(value.startsWith("{")) {
        const jsonValue = JSON.parse(value);
        doc = Node.fromJSON(useSchema, jsonValue);
      }
      else {
        doc = DOMParser.fromSchema(useSchema).parse(appWrapper.parseHTML(value));
      }
    }
    catch(e) {
      console.error("Failed loading ProseMirror content", e, element, value);
    }
  }

  const initialState = EditorState.create({...stateConfig, doc});

  const renderer = appWrapper.getRendererFactory();
  const editableResult = renderer.elements.div();
  const editableElement = editableResult.element;
  element.parentElement!.appendChild(editableElement);

  const view = new EditorView(editableElement, {
    state: initialState,
    nodeViews: {
      media(node, view, getPos) {
        return new MediaView(node, view, getPos, appWrapper);
      }
    },
    editable() { return !element.hasAttribute('disabled') },
    dispatchTransaction(transaction) {
      // Apply the change as-is
      const state = view.state.apply(transaction);

      // Update the view to display the change.
      view.updateState(state);

      // Call the onChange callback if provided
      if (settings.onChange) {
        settings.onChange(state.doc);
      }
    },
  });

  const embedMenuItem = editableElement.querySelector(".ProseMirror-menu-dropdown.menu-embed");
  let embedMenuWrapper: HTMLElement|null = embedMenuItem?.parentElement ?? null;
  while(embedMenuWrapper && !embedMenuWrapper.classList.contains("ProseMirror-menuitem")) {
    embedMenuWrapper = embedMenuWrapper.parentElement;
  }
  if(embedMenuWrapper) {
    embedMenuWrapper.classList.add("pull-menu-item-right");
    embedMenuWrapper.classList.add("highlight-menu-item");
  }

  element.style.display = 'none';

  return { view, editableElement };
}
